#ifndef TNSREMOTEINTERFACEDLL_H_INCLUDED_
#define TNSREMOTEINTERFACEDLL_H_INCLUDED_




/*!
\file
\defgroup MessageSystem Message receiver and messages
\defgroup VersionControl Version Control
\defgroup ReturnValues Return Values
\defgroup USE_LOADLIBRARY LoadLibrary
\defgroup RollBackToMainMenu RollBackToMainMenu: Roll back the dialog to the main menu
\defgroup EndApplication EndApplication: Quit the Application
\defgroup StartNavigation StartNavigation: Start Navigation
\defgroup StartSimulation StartSimulation: Start Simulation
\defgroup LinkMercator2Map LinkMercator2Map: Link a mercator coordinate to a map coordinate
\defgroup SearchAddress SearchAddress: Convert an address string to a mercator position
\defgroup AddStopOffPoint AddStopOffPoint: Add a stop off point to the stationlist
\defgroup AddStopOffPoint2FavoriteList AddStopOffPoint2FavoriteList: Add a stop off point to the favoritelist
\defgroup AddStopOffPoint2PreviousDestinationList AddStopOffPoint2PreviousDestinationList: Add a stop off point to the previous destinationlist
\defgroup GetStopOffPointList GetStopOffPointList: Get all stop off points from the stationlist
\defgroup GetFavoriteList GetFavoriteList: Get all stop off points from the favoritelist
\defgroup ClearStopOffPoint ClearStopOffPoint: Delete a stop off point from the stationlist
\defgroup ClearStopOffPointFromFavoriteList ClearStopOffPointFromFavoriteList: Delete a stop off point from the favoritelist
\defgroup GetPreviousDestinationList GetPreviousDestinationList: Get all stop off points from the previous destinationlist
\defgroup ActualApplicationState ActualApplicationState: Get the actual state of the application
\defgroup SystemInfo SystemInfo: Get information from the system
\defgroup SwitchMap SwitchMap: Change the map
\defgroup ChangeApplicationOptions ChangeApplicationOptions: Change the application options
\defgroup StartGPSRecord StartGPSRecord: Start GPS recording
\defgroup StopGPSRecord StopGPSRecord: Stop GPS recording
\defgroup ShowMessage ShowMessage: Show a messagebox
\defgroup RoutePlanning RoutePlanning: Calc a route and retrieve route information
\defgroup InverseGeoCoding InverseGeoCoding: Inverse geocode mercator coordinates to an address string
\defgroup GPSInfo GPSInfo: Get GPS information
\defgroup BlockUserInteraction BlockUserInteraction: Block user interaction
\defgroup AllowUserInteraction AllowUserInteraction: Allow user interaction
\defgroup DeleteAllStopOffPoints DeleteAllStopOffPoints: Delete all stop off points
\defgroup AddStopOffPoint_CurrentPosition AddStopOffPoint_CurrentPosition: Add a stop off point from the current position to the stationlist
\defgroup ClearStopOffPointFromPreviousDestinationList ClearStopOffPointFromPreviousDestinationList: Delete stop off point from previous destinationlist
\defgroup RefreshMap RefreshMap: Disable/Enable map update
\defgroup GPSLog GPSLog: send NMEA Data to the navigator
\defgroup NaviControls NaviControls: Zoom in/out or skip station
\defgroup SetMapCenter SetMapCenter: Set the map center
\defgroup VisibleMapRect VisibleMapRect: Set map rect
\defgroup DataUpdate DataUpdate: Recalculates the route
\defgroup RiDump RIDump: Dump RI-commands to a file
\defgroup SetProfile SetProfile: Set current car profile
\defgroup GetProfile GetProfile: Get current car profile
\defgroup SetDayNight SetDayNight: Set day/night mode
\defgroup GetDayNight GetDayNight: Get day/night mode 
\defgroup StartNavigationOverBCR StartNavigationOverBCR: Start a navigation from a BCR File
\defgroup CreateBuddyTracking CreateBuddyTracking: create a buddy tracking icon
\defgroup SetBuddyPosition SetBuddyPosition: set the position for a buddy tracking icon
\defgroup DeleteBuddyTracking DeleteBuddyTracking: delete a buddy tracking icon
\defgroup CurrentFuelConsumption CurrentFuelConsumption: Set the current fuel consumption
\defgroup ReRoutingOccurred ReRoutingOccurred: returns the number of rerouting since the last call of the function

*/
/*!

\mainpage notitle
\section mainPageIntro Introduction
The Remote Interface is a set of commands that lets you interact with the fleet / truck navigator software.<br>
It provides over 40 functions in 5 categories, like getting GPS-information, adding stations,
setting system properties or starting a navigation.<br>
These functions are described in detail on the following pages.<br>
To get started with the interface take a look at the \ref firststeps and the \ref tutorials page. They contain guides
from basic to advanced topics.<br>
For further information, we also have some demos on the \ref demos page.



<table cellpadding="2" cellspacing="1" border="0" width="100%" bgcolor="#e5e5e5">
<tr>
<th bgcolor="#cfcec9"  width="33%">Getting Started</th>
<th bgcolor="#cfcec9"  width="33%">API Reference</th>
<th bgcolor="#cfcec9"  width="33%">General</th>
</tr>
<tr>
<td valign="top">
<ul>
<li>\ref firststeps "First Steps"</li>
<li>\ref tutorials</li>
<li>\ref demos</li>
<li>\ref faq</li>
</ul>
</td>
<td valign="top">
<ul>
<li>\ref allevents "All Commands"</li>
<li>\ref GeneralEvents "General Commands"</li>
<li>\ref NavigationEvents "Navigation Commands"</li>
<li>\ref CoordinateRelatedEvents "Coordinate Related Commands"</li>
<li>\ref StationManagement "Station Management Commands"</li>
<li>\ref GPSEvents "GPS Related Commands"</li>
<li><a class="el" href="group___return_values.html">ErrorCodes</a></li>
</ul>
</td>
<td valign="top">
<ul>
<li><a class="el" href="group___version_control.html">Version Control</a></li>
<li>\ref DotNetWrapper ".NET-Wrapper"</li>
</ul>
</td>
</tr>
</table>
*/

//-----------------------------------------------------------


/** \page AboutReleaseNotes Release Notes

  \section ChangeLog ChangeLog of the Remote Interface
  \n
  This is the detailled ChangeLog. 

  [History for earlier versions was removed]

*/

/** \page faq Frequently Asked Questions (FAQs)

 \section FAQ_sec0 General Questions

<B>Q: The API I use fails and I don't know why?</B>
\n
<B>A:</B> Please try to interpret the error code at first! The Errorcodes have all unique meanings and
are mapped to symbols with a somewhat descriptive name.
Then consult the documentation and read about requirements and pre-conditions for that API. Check out the 
supplied example code which demonstrates many calls in a working example.

<B>Q: What about C# support?</B>
\n
<B>A:</B> The remote interface has a C# wrapper. You will need the .NET Framework 2 to use the wrapper, other frameworks are not supported.

<B>Q: Is the interface synchronous?</B>
\n
<B>A:</B> No, the interface is not synchronous, because it uses the windows messaging system to communicate. You have to explicitely check for the acknowledgement message sent from the navigator to your WindowProc()-method to know if the request you sent was processed correctly.

<B>Q: Do I have to wait for the acknowledgement message or can I ignore it if I don't need the return value sent with the message ?</B>
\n
<B>A:</B> You always have to check the acknowledgement message. If you don't check it, you will not know if the command succeeded. Also consecutive commands may fail too because they rely on the correct execution of the previous command.

<B>Q: Is the return value of the C#-Method PostMessage() an indicator for successful execution of the command?</B>
\n
<B>A:</B> No, the return value indicates only that the request was sent. To know if the request succeeded, implement  RIDataArrived() and check for the message you sent.

<B>Q: Why can't SearchAddress geocode the address-string?</B>
\n
<B>A:</B> Search Address works only on the current map. If the address is in another country, 
and the navigation software is in the navigation mode we can't change the map.

<B>Q: Why can't Inverse GeoCoding geocode the coordinates?</B>
\n
<B>A:</B> Inverse GeoCoding  works only on the current map. If the coordinate is in another country, 
and the navigation software is in the navigation mode we can't change the map.

<B>Q: Why are the application options not active, although I set them with ChangeApplicationOptions()?</B>
\n
<B>A:</B> The options must be set before a route calculation. If the options are set 
during the navigation, they will be not taken into account before the next navigation.

<B>Q: Where do I have to install my application to communicate with the navigator?</B>
\n
<B>A:</B> The application can be installed anywhere. To ensure that the application is able to 
communicate with the remote interface, the ri.dll has to be moved to the SYSTEMROOT directory or you must add the the application directory of the navigator
to the $PATH environment variable.

<B>Q: How can I start a navigation with more than one station?</B>
\n
<B>A:</B> The stations must be added individually by AddStopOffPoint, then you can call StartNavigation.

<B>Q: When a navigation is running, how can I start a new navigation?</B>
\n
<B>A:</B> First, you must cancel the current navigation with "RollBackToMainMenu". Then you can start a new navigation.

<B>Q: How can I geocode a WGS84 coordinate to a mercator coordinate?</B>
\n
<B>A:</B>The command "SearchAddress" can be used to geocode WGS84 coordinates. 
The search string has to be in the following format: "-geodecimal(8.418774,49.009233)"
*/

/*!
\page firststeps First Steps

\section FirstStepsIntro Introduction
On this page we will describe the principles of including the remote interface into your application 
and show you how to communicate with it.<br>

\section FirstStepsBasics Basics
The remote interface communicates with the client via window messages, so you should be familiar with these.<br>
The navigation software acts as a server and your application is a client who "talks" to this server.
You can send and receive messages to and from the navigator and there is the possibility to exchange data between 
the applications (the data will be transfered over shared memory).<br>

<h3>What you need</h3>
\li A basic knowledge of window messages
\li A PTV navigator with the remote interface dll (ri.dll)
\li A development environment (eVC4 / Visual Studio).

<h3>What you can do</h3>
\li Change Application options (like the speaker volume)
\li Geocode and inverse geocode addresses and coordinates
\li Control the program flow (like starting a navigation)
\li Handle stations (includes adding and removing of stations)
\li Get GPS information

\section FirstStepsInclude Including the remote interface
To use the interface, add the following to your code:
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"
\endcode
The USE_LOADLIBRARY define causes the dll to be loaded automatically, no other steps will be needed to start.

\section FirstStepsWhat What are commands
Throughout the help, we use the term command for the process of sending the navigator information and getting 
information from the navigator.<br>
Because of the message based system of the interface, it's not a function or a method like in other interfaces, but
a combination of a message and optional data which will belong to this message.<br>
These messages are standard Windows Messages which will be handled in a message loop and therefore the interface is not synchronous. Be sure to have this in mind when sending commands to the navigator !<br>
Every command has a distinct message to send, and some have optional functions to read or write the data.<br>
You will find an overview and further explanation of all commands in \ref allevents "All Commands".

\section FirstStepsSend Sending and receiving commands
First, be sure that the navigator software is running and you are at least in the main menu (see picture below) of the navigator.
<table cellpadding="3" cellspacing="3" border="0" width="100%" bgcolor="#FFFFFF">
<tr>
<td>
\image html pics\mainmenu.jpg
</td>
</tr>
</table>

The procedure of sending commands and receiving the acknowledgement answer is always the same:<br>
\image html pics\ri_processing.jpg
<p>
You send a message to the interface (e.g. "RI_MESSAGE_SYSTEMINFO" to get some system information). To know if the request succeeded, check in your WindowProc()-procedure for the same message you sent (the navigator will sent this message back after executing the request). When the message arrives, the wParam of WindowProc() will be RI_NOERROR on success or an error value if the request failed.<br>
<b>It is essential to check the acknowledgement message in the WindowProc to know if the command succeeded.</b><p>
The messages you can send are described in the Commands-Description (see \ref allevents "All commands").<p>
Before sending a message to the remote interface, you have to take care that the interface is active and loaded.<br>
This is done by the method RI_GetTNS() which returns a window-handle to the interface.<p>
In the \ref tutorials, you will find this code to check for the interface:
\code
if ( !IsWindow( RI_GetTNS() ) )
                return RI_NAVIGATIONNOTACTIVE;
\endcode

With this handle, you can use the PostMessage() method to send the message to the interface:
\code
PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
\endcode
Each message sent must have a unique identifier, see parameter 'id' in the above example.<p>
For the tutorials and for general use, we wrote two little convenience functions:
\li RI_MESSAGE which will check for the handle and send the message, so you don't have to do it for yourself, and
\li GetUniqueID which will create the appropriate id.<br>
Both methods are used in the tutorials.

\section FirstStepsTypes Types of commands
There are two types of RI-commands:
\li Commands without data-transfer (like setting the day/night mode)
\li Commands with data-transfer (like adding a station).

Commands without data-transfer are the simplest way to communicate with the interface. Just use the above mentioned 
RI_MESSAGE function to send the request to the navigator, check in WindowProc() if the message succeeded (the navigator will always send a message back to your app to signalize the proper processing of the request and for error handling) and your done.<p>

Commands with data-transfer can send and receive data from the interface. For this purpose, every command with data-transfer capability has it's own functions to read and write the data.<br>
As an example, the read-function for the SearchAddress-Command is called "RI_SearchAddress_ReadData()", 
the write-function is called "RI_SearchAddress_WriteData()".<br>
The data for the transfer is stored in a struct, in the SearchAddress example it's called "RI_CSearchAddress".<br>
(See \ref allevents "All Commands" for details).<p>
To send data, first fill the struct with the appropriate content, then write the data to the interface and at last, send
the command:
\code
RI_CSearchAddress data;
wcscpy(data.m_address, L"D,76131,Karlsruhe,Stumpfstrasse,1");

LPARAM id = GetUniqueID();
LRESULT sharing_ret = RI_SearchAddress_WriteData( id, data );

if ( RI_MESSAGE( RI_MESSAGE_SEARCHADDRESS, GetSafeHwnd(), id ) == RI_NOERROR )
{
}
\endcode
Again, you have to create a unique id. This time, you must use the same id for writing the data and sending the message (the id
is needed to relate the data to the message).<p>

To receive data, check in wndproc for your message and read the data with the reader-function:
\code
if ( message == RI_MESSAGE_SEARCHADDRESS )
{
    if ( (LRESULT)wParam == RI_NOERROR )
    {
        RI_CSearchAddress data;
        LRESULT read_suc = RI_SearchAddress_ReadData( lParam, data );
    }
}
\endcode

\section FirstStepsRestrictions Restrictions when sending commands
<b>Avoid mixing remote interface calls with user interaction, this can result in undefined behaviour.</b><p>
A problem that sometimes occurs is that the client sends a remote command and the user of the navigation software presses
a button (e.g. sending a station will automatically switch to the stationlist dialog, and some users press the
"calculate" button while the navigator is adding some other station).
To avoid these situations the best way is to disable the user interface by the client before sending such 
commands and enable it afterwards. If the client doesn't block the user interaction, the remote interface will
autoblock it for certain commands.<br>

These commands are:

\li \ref StartNavigation
\li \ref StartNavigationOverBCR
\li \ref StartSimulation
\li \ref RoutePlanning

In these routines, the remote interface will disable user interaction with the navigator before executing 
the call and enable it after the call, if there was no explicit BlockUserInteraction-command sent by the client.<br>
This will not help in situations like adding multiple stations to the navigator. In this case, if you don't block
the interface, we block it, but we can do this only an a command basis, so the result would be:<br>
block interface -> add station -> unblock -> block -> addstation -> unblock and so on.<br>
This is most time not what you expect and doesn't avoid all user interaction.<p>
<b>We recommend to explicitly block the user interface always by calling BlockUserInteraction before sending commands 
where the user may have interaction possibilities.</b><p>
Calling BlockUserInteraction will disable the internal block/allow mechanism.<p>

\section FirstStepsNext Next steps
To find out more, see the \ref tutorials page.
*/

/*!
\page tutorials Tutorials
\section TutorialsIntro Introduction
The tutorials will show you how to integrate the remote interface in your application 
and how to communicate with the navigator.<br>
As described in \ref firststeps there are two types of commands, the ones without data and the others with data-transfer.
The commands with data-transfer can be splitted in two subgroups:
\li commands with one package of data received (this type was introduced in \ref firststeps)
\li commands with multipackage data return.<br>

For every type of command, we have created a small tutorial to show how the control flow works.<br>
We also have two convenience functions which will be used in our examples:<p>
Every message needs a unique id to be sent, so we create one by this function:
\code
LPARAM GetUniqueID()
{
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}
\endcode

Furthermore we need a function to get a handle from the remote interface and send the request to it:
\code
inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    if ( !IsWindow( RI_GetTNS() ) )
    return RI_NAVIGATIONNOTACTIVE;
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}
\endcode
The return value of the above function is set to RI_NAVIGATIONNOTACTIVE or RI_NOERROR only to check if the command itself could be posted, it does not indicate that the request to the interface was successful. To check for successful execution of the command wait for the message you sent in the WindowProc to arrive.<p>
\section TutorialsStartGetStart Tutorials
\ref StartTracking "Tutorial 1:" Start the navigation software in tracking mode (command without data-transfer)<br>
\ref GeoCoding "Tutorial 2:" Convert an address-string to mercator coordinates (command with data-transfer, single package)<br>
\ref ReadStationList "Tutorial 3:" Read the stationlist from the navigator (command with data-transfer, multiple package)<br>
*/



/*!
\page StartTracking Start Tracking Mode
\section StartTrackingIntro Introduction
This Example shows how to switch the navigator into tracking mode.<br>
We do not need to send data for this request, 
so the only thing we do is sending the correspondent message:
\code
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_STARTNAVIGATION, GetSafeHwnd(), id ) == RI_NOERROR )
    {
    }
\endcode
This will cause the navigation software to switch to the tracking mode.<br>
In the WindowProc-function, we wait for the answer. The navigator will send the same message back and sets the wParam to the error code.<br> 
In this case, if wParam = RI_NOTADDED, the navigator switched successfully to the tracking mode:
\code
LRESULT CDlgMFC_StartTracking::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    if ( message == RI_MESSAGE_STARTNAVIGATION )
    {
        if ( (LRESULT)wParam == RI_NOTADDED )
        {
            //succeeded
        }
    .
    .
    .
    }
}
\endcode

<h3>The complete sequence:</h3>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) )
        return RI_NAVIGATIONNOTACTIVE;
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_StartTracking::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_STARTNAVIGATION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Navigation succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_USER)
            //Navigation not succeeded, User abort Navigation. errorcode (20)
        else if ( (LRESULT)wParam == RI_NOTADDED)
            //Tracking Modus succeeded. errorcode (11)
        else if ( (LRESULT)wParam == RI_NOFIX)
            //Navigation not succeeded, no GPS Fix. errorcode (18)
        else if ( (LRESULT)wParam == RI_NOTRUN)
            //Navigation not succeeded, Route calculation failed. errorcode (17)
        else if ( (LRESULT)wParam == RI_START_RETRACTED )
            //Navigation succeeded. Start was retracted. errorcode (21)
        else if ( (LRESULT)wParam == RI_TARGET_RETRACTED )
            //Navigation succeeded. Target was retracted. errorcode (22)
        else if ( (LRESULT)wParam == RI_START_TARGET_RETRACTED )
            //Navigation succeeded. Start and Target was retracted.errorcode (23)
        else 
            //Navigation not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_StartTracking::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_STARTNAVIGATION, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}

\endcode
*/

/*!
\page GeoCoding GeoCoding Example
\section GeoCodingIntro Introduction
This Example will show you how to get a coordinate from an address string. This time, we do not only send
a message, we also have to pass data (the address string) to the navigator and we will read the result.
Every command that can read or write data has it's own functions to do this and a struct in which the data is stored.<br>
In case of the SearchAddress-Command they are called:
\li RI_SearchAddress_ReadData()
\li RI_SearchAddress_WriteData()
\li RI_CSearchAddress (the struct)<p>
First, we create and fill the struct with the search string:
\code
    RI_CSearchAddress data;
    wcscpy(data.m_address, L"D,76131,Karlsruhe,Stumpfstrasse,1");
\endcode
Then, we write the data into the shared memory:
\code
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SearchAddress_WriteData( id, data );
\endcode
No we can send the message to search the address.
The id for reading and writing the data the will be also used for RI_MESSAGE:
\code
    if ( RI_MESSAGE( RI_MESSAGE_SEARCHADDRESS, GetSafeHwnd(), id ) == RI_NOERROR )
    {
    }
\endcode
After sending the request, we wait in windowproc for the returned answer.<br>
The wParam-value will indicate the error code.
\code
    if ( message == RI_MESSAGE_SEARCHADDRESS )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CSearchAddress data;
            LRESULT read_suc = RI_SearchAddress_ReadData( lParam, data );
        }
    }
\endcode
If the message is the one we sent, we read the data with RI_SearchAddress_ReadData().<br>
The lParam value is the id we sent before, the struct will be filled with the result.



<h3>The complete sequence:</h3>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GeoCoding::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SEARCHADDRESS )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //SearchAddress succeeded. errorcode (0)
            //get the data from the shared memory
            RI_CSearchAddress data;
            LRESULT read_suc = RI_SearchAddress_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten. errorcode (-2)<BR>
            else
            {
                TRACE(TEXT("X-Koordinate: %i\n"), data.m_mercator_x);
                TRACE(TEXT("Y-Koordinate: %i\n"), data.m_mercator_y);
                TRACE(TEXT("Result Address: %s\n"), data.m_result_address);
            }
        }
        else if ( (LRESULT)wParam == RI_NOTAV )
        {
            //SearchAddress has no street found. errorcode (19)
            //get the data from the shared memory
            RI_CSearchAddress data;
            LRESULT read_suc = RI_SearchAddress_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )<BR>
                //...data was overwritten. errorcode (-2)
            else
            {
                TRACE(TEXT("X-Koordinate: %i\n"), data.m_mercator_x);
                TRACE(TEXT("Y-Koordinate: %i\n"), data.m_mercator_y);
                TRACE(TEXT("Result Address: %s\n"), data.m_result_address);
            }
        }
        else
            //SearchAddress not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GeoCoding::foo()
{
    //*** copy data ***
    RI_CSearchAddress data;
    wcscpy(data.m_address, L"D,76131,Karlsruhe,Stumpfstrasse,1");

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SearchAddress_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_SEARCHADDRESS, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/

/*!
\page ReadStationList Reading a stationlist from the navigator
\section ReadStationListIntro Introduction
In this example we will show how to handle a request with multiple packages returned.
The sending of the request is similar to the tutorials we saw before, so we do not explain it here anymore.<br>
In the windowproc, we check for our message as in the tutorials before:
\code
LRESULT CDlgMFC_ReadStationList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    if ( message == RI_MESSAGE_GETSTOPOFFPOINTLIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CGetStopOffPointList data;
            LRESULT read_suc = RI_GetStopOffPointList_ReadData( lParam, data );
\endcode
In this case, the received a data struct that has a field m_listcount.
This field holds the number of stations available. If you want to read all stations, you have to
increase the m_index field in this received struct, write it with the writer function back to the shared memory and
resend the message.<br> So the next time, we receive the message, the second package will be transfered, and so on 
until there are no more packages available.
\code
    if (data.m_index < data.m_listcount )
    {
        data.m_index = data.m_index + 1;
        LRESULT sharing_ret = RI_GetStopOffPointList_WriteData( data.m_ID, data );
        RI_MESSAGE( RI_MESSAGE_GETSTOPOFFPOINTLIST, GetSafeHwnd(), data.m_ID );
    }
\endcode

<h3>The complete sequence:</h3>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ReadStationList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_GETSTOPOFFPOINTLIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CGetStopOffPointList data;
            LRESULT read_suc = RI_GetStopOffPointList_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten. errorcode (-2)
            else
            {   
                //examine whether the current station-index is smaller than the stationcount
                if (data.m_index < data.m_listcount )
                {
                    //show data:
                    TRACE(TEXT("Caption: %s\n"),data.m_Caption);
                    TRACE(TEXT("Description: %s\n"), data.m_Description);
                    TRACE(TEXT("X-Koorinate: %i\n"), data.m_mercator_x);
                    TRACE(TEXT("Y-Koordinate: %i\n"), data.m_mercator_y);
                    TRACE(TEXT("ID: %i"),data.m_StationID);
                    //advance station counter 1
                    data.m_index = data.m_index + 1;
                    //write data
                    LRESULT sharing_ret = RI_GetStopOffPointList_WriteData( data.m_ID, data );
                    //get next station from the list
                    RI_MESSAGE( RI_MESSAGE_GETSTOPOFFPOINTLIST, GetSafeHwnd(), data.m_ID );
                }
            }

        }
        else 
            //Get Stop of Point List not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ReadStationList::foo()
{
    //*** copy data ***
    RI_CGetStopOffPointList data;
    //write flag and index
    data.m_index = 0;

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_GetStopOffPointList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
	    return;
    if ( RI_MESSAGE( RI_MESSAGE_GETSTOPOFFPOINTLIST, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/


/*!
\page demos Demos
\section DemosIntro Introduction
On this page you can find some examples we could not put in \ref tutorials, because of their size.<br>
The examples are much bigger than the tutorials and will show you how complex operations are done.

<table cellpadding="3" cellspacing="3" border="0" width="100%" bgcolor="#e5e5e5">
<tr>
<td valign="top" width="50%">
<h2>Demo 1 (C++ - Example)</h2>
In this demo, we will decode an address string (get the coordinates), add these coordinates as a new station to the navigator and
finally start a navigation to this point.<p>
<a href="StartNavigation.zip"><b>Download Demo 1</b></a>
</td>
<td width="50%">
\image html pics\Demo1.gif
</td>
</tr>

</table>

<p>
<table cellpadding="3" cellspacing="3" border="0" width="100%" bgcolor="#e5e5e5">
<tr>
<td valign="top" width="50%">
<h2>Demo 2 (.NET - Example)</h2>
This demo provides a little user interface with a listbox and some buttons, which trigger different commands:

\li the gps signal is polled periodically and if we reach the last destination, the button "complete" will be enabled
\li pressing button "Info" will get the stationlist from the navigator and print the stations in the listbox
\li pressing the button "Nv1" will start a navigation with data from a BCR-File
\li pressing button "Nv2" will trigger a chain of commands:<br>
first, the navigator will be set to the main menu,<br>
second, a station will be added to the navigator,<br>
third, a navigation to the added station will be started.<p>
<a href="GPSInfo.zip"><b>Download Demo2</b></a><p>
</td>
<td width="50%">
\image html pics\Demo2.gif
</td>
</tr>

</table>

*/

/*!
\page DotNetWrapper .NET-Wrapper
\section DotnetWrapperIntro Introduction
<b>Please read \ref firststeps to get an overview what commands are and how they work.</b><p>
The .NET-Wrapper translates the remote interface calls into a .NET-compatible interface.<br>
\section DotnetWrapperInclude Including the interface
<b>You will need the .NET Framework 2 to use the wrapper, other Frameworks are not supported.</b><p>
To use the wrapper, you need to add the RiDNWrapper.dll to the references of your project.
Next, you have to include the library into your sources with 
\code
using RiDNWrapper;
\endcode
Now you are ready to use the functions.<br>

\section DotnetWrapperGettingStarted Getting Started

The .NET-Wrapper provides a number of classes to interact with the navigator:
\li IRI is the main class which provides the access to the remote interface.<br>
\li cWndMsgs is the container for all messages the remote interface provides.<br>

Let's start with declaring a member which is of type IRI-class:
\code
    private int m_ID = 0;
    private IRI m_RI; 
\endcode
m_ID is a unique ID needed to communicate with the Remote Interface.<br>
IRI is the main access class to the remote interface functionality, it provides methods to send requests and
data.
The answers to these requests will be received in a method called RIDataArrived() which will be described later on.<br>
We do not instantiate IRI directly but cRI which is derived from IRI and adds some more functionality.<p>
No we create an instance of cRI, and initialize it:
\code
m_RI = new cRI();
m_RI.Init(this);
\endcode
The next step is to get the handle from the remote interface, which we will need in further communications:
\code
int hWnd = m_RI.GetTNS();
if (hWnd == 0)
{
    //Error: Navigator not running
} 
\endcode

\section DotnetWrapperSending Sending and receiving commands
As decribed in the \ref firststeps, the RI-Interface is based on window messages and therefore, the interface is asynchronous. After you have sent a message to the navigator (with or without data transfer) the navigator will process the request and send back an acknowledgement message to your client after he finished his work. You can only be sure that a command succeeded if you check this acknowledgement message in RiDataArrived() ! <p>
There are two different types of commands, the ones without data-transfer and the other ones with data-transfer.<br>
Sending a command without data-transfer is done like this (we will send the command to set the navigator back to 
the main menu):<br>
\code
m_ID++;
int iRet = m_RI.PostMessage(cWndMsgs.MESSAGE_ROLLBACKTOMAINMENU, m_ID);
if (iRet != (int)RI_Errors.RI_NOERROR)
{
}
\endcode
Initially, we increase m_ID to get a unique number. Then we call the method cRI::PostMessage() with the appropriate parameters (the message you want to send, and the id). This is pretty much the same as in C++ .<br>

Sending a command with data-transfer:<br>
The data to be transfered will be handled by the class which corresponds to the message.
Before sending the command we have to create a new instance of the needed class and fill the members of it appropriately (in our example, we will send a request to get the list of stop-off-points):<br>
\code
cRI_CGetStopOffPointList data = new cRI_CGetStopOffPointList();
data.m_index = 0;
\endcode
We want the first entry of the list, so we set m_index to 0.
\code
int iRet = m_RI.GetStopOffPointList_WriteData(m_ID, data);
\endcode
Now we write the data to the remote interface. This will send the message automatically (all writer functions will do 
this for us, no expicit call to PostMessage() is needed).<br>

\section DotnetWrapperReceiving Receiving messages and data
To receive messages and data, we have to implement the method RIDataArrived():
\code
public void RIDataArrived(int RIMessage, object data, int ReturnValue)
{
}
\endcode
The handling in this method is similar to the handling in the wndproc method in C++, check for the message you
previously sent and handle it:
\code
if (RIMessage == cWndMsgs.MESSAGE_GETSTOPOFFPOINTLIST)
{
	if (ReturnValue != (int)RI_Errors.RI_NOERROR)
	{
		// error handling...
	}
	else if (data is cRI_CGetStopOffPointList)
	{
		cRI_CGetStopOffPointList a = (cRI_CGetStopOffPointList)data;
		if (a.m_ID == m_ID)
		{
			if (a.m_index < a.m_listcount)
			{
				// access data...
			}
		}
	}
}
\endcode
It is important that you call the Reader-Function after the navigator sent back the acknowledgement message, otherwise, it is not guaranteed that the data is complete and correct.<br>

For further information, please see \ref demos and <a href="RI-Wrapper.zip"> RI-Wrapper.zip</a>
*/


/*!
\page allevents All Commands
\section AllEventsIntro Introduction
This is an overview of all commands available.<br>

\section General
\li \ref ActualApplicationState
\li \ref SystemInfo
\li \ref SwitchMap
\li \ref ChangeApplicationOptions
\li \ref ShowMessage
\li \ref BlockUserInteraction
\li \ref AllowUserInteraction
\li \ref SetProfile
\li \ref GetProfile
\li \ref SetDayNight
\li \ref GetDayNight
\li \ref RollBackToMainMenu
\li \ref EndApplication
\li \ref CurrentFuelConsumption
\section Navigation
\li \ref StartNavigation
\li \ref StartSimulation
\li \ref RoutePlanning
\li \ref RefreshMap
\li \ref NaviControls
\li \ref SetMapCenter
\li \ref VisibleMapRect
\li \ref DataUpdate
\li \ref StartNavigationOverBCR
\li \ref CreateBuddyTracking
\li \ref SetBuddyPosition
\li \ref DeleteBuddyTracking
\li \ref ReRoutingOccurred
\section Coordinates
\li \ref LinkMercator2Map
\li \ref SearchAddress
\li \ref InverseGeoCoding
\section Stations
\li \ref AddStopOffPoint
\li \ref AddStopOffPoint2FavoriteList
\li \ref AddStopOffPoint2PreviousDestinationList
\li \ref GetStopOffPointList
\li \ref GetFavoriteList
\li \ref ClearStopOffPoint
\li \ref ClearStopOffPointFromFavoriteList
\li \ref GetPreviousDestinationList
\li \ref DeleteAllStopOffPoints
\li \ref AddStopOffPoint_CurrentPosition
\li \ref ClearStopOffPointFromPreviousDestinationList
\section GPS
\li \ref StartGPSRecord
\li \ref StopGPSRecord
\li \ref GPSInfo
\li \ref GPSLog
*/

/*!
\page GeneralEvents General Commands
\li \ref ActualApplicationState
\li \ref SystemInfo
\li \ref SwitchMap
\li \ref ChangeApplicationOptions
\li \ref ShowMessage
\li \ref BlockUserInteraction
\li \ref AllowUserInteraction
\li \ref SetProfile
\li \ref GetProfile
\li \ref SetDayNight
\li \ref GetDayNight
\li \ref RollBackToMainMenu
\li \ref EndApplication
\li \ref CurrentFuelConsumption
*/

/*!
\page NavigationEvents Navigation Commands
\li \ref StartNavigation
\li \ref StartSimulation
\li \ref RoutePlanning
\li \ref RefreshMap
\li \ref NaviControls
\li \ref SetMapCenter
\li \ref VisibleMapRect
\li \ref DataUpdate
\li \ref StartNavigationOverBCR
\li \ref CreateBuddyTracking
\li \ref SetBuddyPosition
\li \ref DeleteBuddyTracking
\li \ref ReRoutingOccurred
*/

/*!
\page CoordinateRelatedEvents Coordinate Related Commands
\li \ref LinkMercator2Map (obsolete, will be removed in later versions)
\li \ref SearchAddress
\li \ref InverseGeoCoding
*/

/*!
\page StationManagement Station Management Commands
\li \ref AddStopOffPoint
\li \ref AddStopOffPoint2FavoriteList
\li \ref AddStopOffPoint2PreviousDestinationList
\li \ref GetStopOffPointList
\li \ref GetFavoriteList
\li \ref ClearStopOffPoint
\li \ref ClearStopOffPointFromFavoriteList
\li \ref GetPreviousDestinationList
\li \ref DeleteAllStopOffPoints
\li \ref AddStopOffPoint_CurrentPosition
\li \ref ClearStopOffPointFromPreviousDestinationList
*/

/*!
\page GPSEvents GPS Related Commands
\li \ref StartGPSRecord
\li \ref StopGPSRecord
\li \ref GPSInfo
\li \ref GPSLog
\li \ref GetFavoriteList
\li \ref ClearStopOffPoint
\li \ref ClearStopOffPointFromFavoriteList
\li \ref GetPreviousDestinationList
\li \ref DeleteAllStopOffPoints
\li \ref AddStopOffPoint_CurrentPosition
\li \ref ClearStopOffPointFromPreviousDestinationList
*/

// no windows headers are included as the application context is not known
#ifndef UNDER_CE
#include <cstdlib>   // for wcstombs
#include <cassert>
#endif

// *********************************************************************************
//  Definition stuff
// ********************************************************************************

#ifdef DLLTNSRI_EXPORTS
#define DLLTNSRI_API __declspec(dllexport)
#else
#define DLLTNSRI_API __declspec(dllimport)
#endif


/*! \namespace */ 
namespace {

	const unsigned long m_RI_DLLHeaderVersion = 910; // 909 = 00.09.10 = 0.9.10
}

/*!
	\ingroup VersionControl
	\brief Get the version number of the header.
	\return m_RI_DLLHeaderVersion (header version number)

	Both, the dll and the header have a version number.
	Check the version number to avoid problems with different versions.
	The version number has 6 digits xx.xx.xx.

	- 0.9.10	=    910 : Initial version number
*/
inline unsigned long RI_GetDLLHeaderVersion()
{
	return m_RI_DLLHeaderVersion;
}



/*!
    \internal
	\ingroup USE_LOADLIBRARY
	If USE_LOADLIBRARY is set, the header file automatically loads the dll when 
	the first call to a function is carried out.
	Even if this is quite comfortable, the dll should be loaded as early as possible. 
	Not loadind the dll early might affect serious fragmentation in the 
	navigators virtual memory.

	To force the dll-loading early, just check the dll-version :). This should be done anyway.

	Do not mix file includes using USE_LOADLIBRAY and not using LOADLIBRARY! If you need
	to incude the header more than once, think of using a compiler-wide define.
*/
#ifdef USE_LOADLIBRARY

#if defined(USE_LOADLIBRARY) && defined(DLLTNSRI_EXPORTS)
#error compiler-error? true. the flag combination is invalid.
#endif

/*!
	As Win32 does not have a unicode compatible GetProcAddress-Function
	we use this indirection.
	In the Win32-version a conversion to char is carried out if necessary.
*/
#ifdef UNDER_CE
inline FARPROC InternalGetProcAddress(HMODULE hModule, const wchar_t * lpProcName)
{
	return GetProcAddress(hModule, lpProcName);
}
#else // UNDER_CE
inline FARPROC InternalGetProcAddress(HMODULE hModule, const wchar_t * lpProcName)
{
	size_t lpProcNamenLen = wcslen(lpProcName);
	char dummy[256];
	wcstombs( dummy, lpProcName, 256);
	dummy[lpProcNamenLen]=0;
	FARPROC retVal = GetProcAddress(hModule, dummy);
	return retVal;
}
#endif // UNDER_CE

//
// The DLL_FUNC_x macros are used to distinct between USE_LOADLIBRARY and !USE_LOADLIBRARY
//
// USE_LOADLIBRARY:
//   DLL_FUNC_x defines a function NAME, and tries to load the correponding function
//   from the dll.
//   \return if not successful it will return DEFAULT_RETURN (of the type RETURN)
//           if successful it will return the return value of the function
//   \param All parameters are passed to the function. The type has to be defined (PARAM_x)
//
// !USE_LOADLIBRARY
//   DLL_FUNC_x declares the function NAME
//
#define DLL_FUNC_0( NAME, RETURN, DEFAULT_RETURN ) \
typedef __declspec(dllimport) RETURN ( * p##NAME )(); \
inline RETURN __stdcall NAME() \
{ \
	static p##NAME function_pointer( (p##NAME)InternalGetProcAddress( CRemoteDLL::getDLL(), TEXT(#NAME) ) ); \
	if ( function_pointer ) return function_pointer(); \
	else return DEFAULT_RETURN; \
}

#define DLL_FUNC_1( NAME, RETURN, DEFAULT_RETURN, PARAM_1 ) \
typedef __declspec(dllimport) RETURN ( * p##NAME )( PARAM_1 ); \
inline RETURN __stdcall NAME( PARAM_1 param_1 ) \
{ \
	static p##NAME function_pointer( (p##NAME)InternalGetProcAddress( CRemoteDLL::getDLL(), TEXT(#NAME) ) ); \
	if ( function_pointer ) return function_pointer( param_1 ); \
	else return DEFAULT_RETURN; \
}

#define DLL_FUNC_2( NAME, RETURN, DEFAULT_RETURN, PARAM_1, PARAM_2 ) \
typedef __declspec(dllimport) RETURN ( * p##NAME )( PARAM_1, PARAM_2 ); \
inline RETURN __stdcall NAME( PARAM_1 param_1, PARAM_2 param_2 ) \
{ \
	static p##NAME function_pointer( (p##NAME)InternalGetProcAddress( CRemoteDLL::getDLL(), TEXT(#NAME) ) ); \
	if ( function_pointer ) return function_pointer( param_1, param_2 ); \
	else return DEFAULT_RETURN; \
}

#else // USE_LOADLIBRARY

#if !defined(NDEBUG) && defined(DLLTNSRI_EXPORTS) && (_MSC_VER > 1300)
// In this block additional checks are carried out, when the dll is 
// built in debug mode. 


// PK: we define simple traits to convert references types to value types
template < typename T > class type_traits
	{ public: typedef T value_type; };
template < typename T > class type_traits< T & >
	{ public: typedef T value_type; };

// we add a funtion calling the declared function. if the function is not defined,
// we get a linking error.
#define DLL_FUNC_0( NAME, RETURN, DEFAULT_RETURN ) \
	DLLTNSRI_API RETURN NAME(); \
	void LINK_TEST_##NAME() { \
		NAME(); }
#define DLL_FUNC_1( NAME, RETURN, DEFAULT_RETURN, PARAM_1 ) \
	DLLTNSRI_API RETURN NAME( PARAM_1 ); \
	void LINK_TEST_##NAME() { \
		type_traits<PARAM_1>::value_type param_1 = type_traits<PARAM_1>::value_type(); \
		NAME(param_1); }
#define DLL_FUNC_2( NAME, RETURN, DEFAULT_RETURN, PARAM_1, PARAM_2 ) \
	DLLTNSRI_API RETURN NAME( PARAM_1, PARAM_2 ); \
	void LINK_TEST_##NAME() { \
		type_traits<PARAM_1>::value_type param_1 = type_traits<PARAM_1>::value_type(); \
		type_traits<PARAM_2>::value_type param_2 = type_traits<PARAM_2>::value_type(); \
		NAME(param_1,param_2); }

#else // !defined(NDEBUG) && defined(DLLTNSRI_EXPORTS)

#define DLL_FUNC_0( NAME, RETURN, DEFAULT_RETURN ) \
	DLLTNSRI_API RETURN NAME();
#define DLL_FUNC_1( NAME, RETURN, DEFAULT_RETURN, PARAM_1 ) \
	DLLTNSRI_API RETURN NAME( PARAM_1 );
#define DLL_FUNC_2( NAME, RETURN, DEFAULT_RETURN, PARAM_1, PARAM_2 ) \
	DLLTNSRI_API RETURN NAME( PARAM_1, PARAM_2 );

#endif // !defined(NDEBUG) && defined(DLLTNSRI_EXPORTS)

#endif // USE_LOADLIBRARY

//
// The REQ_FUNC_x macros are specializations of DLL_FUNC_x.
// Nearly all of the requests are defined this way.
//
// RETURN is always LRESULT
// DEFALUT_RETURN is always RI_FUNCTIONNOTLOADED
// PARAM_1 is always LPARAM
//
#define REQ_FUNC_0( NAME ) \
	DLL_FUNC_0( NAME, LRESULT, RI_FUNCTIONNOTLOADED )
#define REQ_FUNC_1( NAME ) \
	DLL_FUNC_1( NAME, LRESULT, RI_FUNCTIONNOTLOADED, LPARAM )
#define REQ_FUNC_2( NAME, PARAM_2 ) \
	DLL_FUNC_2( NAME, LRESULT, RI_FUNCTIONNOTLOADED, LPARAM, PARAM_2 )


// *********************************************************************************
//  Messages
// ********************************************************************************

/*!
    \internal
	\ingroup MessageSystem
	\brief Macro to register a named message.
	
	Register a named window message valid across all applications.
	Name should be clumsy to avoid naming problems.
*/
#define REGISTER_RIMESSAGE( NAME ) \
	const UINT NAME = RegisterWindowMessage(TEXT(#NAME))

/*! \namespace */ 
namespace {
/** @ingroup RollBackToMainMenu
 *  Message ID for the Request "roll back to main menu":<BR>
	RI_MESSAGE_ROLLBACKTOMAINMENU

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for RollBackToMainMenu:</H3>
	RI_NOERROR = Roll Back to Main Menu succeeded.<BR>
	RI_NODESC = Roll Back to Main Menu not succeeded.<BR>
	
	<h2>Description:</h2> Rolls back the navigator to the main menu. Can be used to cancel a navigation or
    a simulation.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_RollBackToMainMenu::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_ROLLBACKTOMAINMENU )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Roll Back to Main Menu succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC)
            //Roll Back to Main Menu not succeeded (error 12)
        else 
            //Roll Back to Main Menu not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_RollBackToMainMenu::foo()
{
    LPARAM id = GetUniqueID();
    //send the message to RI
    if ( RI_MESSAGE( RI_MESSAGE_ROLLBACKTOMAINMENU, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ROLLBACKTOMAINMENU );					//...roll back to main menu
/** @ingroup BlockUserInteraction
 *  Message ID for the Request "block user interaction":<BR>
	RI_MESSAGE_BLOCKUSERINTERACTION

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for BlockUserInteraction:</H3>
	RI_NOERROR = BlockUserInteraction succeeded. errorcode (0)<BR>
	
	<h2>Description:</h2>Disables the user interaction (mouse and key) with the navigator.<br>
    To enable the user input, see \ref AllowUserInteraction "AllowUserInteraction".
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_BlockUserInteraction::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_BLOCKUSERINTERACTION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //BlockUserInteraction succeeded. errorcode (0)
        else 
            //BlockUserInteraction not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_BlockUserInteraction::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_BLOCKUSERINTERACTION, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_BLOCKUSERINTERACTION );
/** @ingroup AllowUserInteraction
 *  Message ID for the Request "allow user interaction":<BR>
	RI_MESSAGE_ALLOWUSERINTERACTION

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for AllowUserInteraction:</H3>
	RI_NOERROR = AllowUserInteraction succeeded. errorcode (0)<BR>
	
	<h2>Description:</h2>Enables the user interaction (mouse and key) with the navigator.<br>
    To disabe the user input, see \ref BlockUserInteraction "BlockUserInteraction".
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_AllowUserInteraction::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_ALLOWUSERINTERACTION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //AllowUserInteraction succeeded. errorcode (0)
        else
            //AllowUserInteraction not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_AllowUserInteraction::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_ALLOWUSERINTERACTION, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ALLOWUSERINTERACTION );
/** @ingroup LinkMercator2Map
 *  Message ID for the Request "link mercator to map":<BR>
	RI_MESSAGE_LinkMercator2Map

	<H3>Write Memory:</H3>
	RI_LinkMercator2Map_WriteData(LPARAM id, RI_CLinkMercator2Map data );
	<BR>
	<H3>Read Memory:</H3>
	RI_LinkMercator2Map_ReadData(LPARAM id, RI_CLinkMercator2Map data );
	<BR>
	<H3>Return Values for LinkMercator2Map:</H3>
	RI_NOERROR = Link Mercator to Map succeeded<BR>
	RI_NOTLINKABLE = the coordinates are not linkable<BR><BR>

	<h2>Description:</h2> Formerly this call matched the pos set by RI_LinkMercator2Map_WriteData()
	to a street.<br>In the current implementation, RI_LinkMercator2Map_ReadData() returns the pos
	set by the writer.
    <h3>Returns a single package.</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_LinkMercator2Map::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_LinkMercator2Map )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //LinkMercator2Map succeeded. errorcode (0)
            //get the data from the shared memory
            RI_CLinkMercator2Map data;
            LRESULT read_suc = RI_LinkMercator2Map_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten. errorcode (-2)
            else
            {
                TRACE(TEXT("X-Koordinate: %i\n"), data.m_linked_mercator_x);
                TRACE(TEXT("Y-Koordinate: %i\n"), data.m_linked_mercator_y);
            }
        }
        else if ( (LRESULT)wParam == RI_NOTLINKABLE )
        {
            //LinkMercator2Map is not linkable. errorcode (10)
        }
        else
            //LinkMercator2Map not succeeded. unkown error
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_LinkMercator2Map::foo()
{
    //*** copy data ***
    RI_CLinkMercator2Map data;
    data.m_mercator_x = 935040;
    data.m_mercator_y = 6269625;

    //*** write data to shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_LinkMercator2Map_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_LinkMercator2Map, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_LINKMERCATOR2MAP );
/** @ingroup SearchAddress
 *  Message ID for the Request "search address":<BR>
	RI_MESSAGE_SEARCHADDRESS

	<H3>Write Memory:</H3>
	RI_SearchAddress_WriteData(LPARAM id, RI_CSearchAddress data );
	<BR>
	<H3>Read Memory:</H3>
	RI_SearchAddress_ReadData(LPARAM id, RI_CSearchAddress data );
	<BR>
	<H3>Return Values for SearchAddress:</H3>
	RI_NOERROR = Search address succeeded<BR>
	RI_NOTAV   = SearchAddress no street found<BR>
	RI_NOTFOUND = No Address found<BR><BR>

	<h2>Description:</h2>Converts an address-string to a mercator position. If the return value is
    "RI_NOTAV", the returned coordinates will be the center of the town (for more details, see
	description in data structure).
    <h3>Returns a single package.</h3>

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_SearchAddress::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SEARCHADDRESS )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //SearchAddress succeeded. errorcode (0)
            //get the data from the shared memory
            RI_CSearchAddress data;
            LRESULT read_suc = RI_SearchAddress_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten. errorcode (-2)<BR>
            else
            {
                TRACE(TEXT("X-Koordinate: %i\n"), data.m_mercator_x);
                TRACE(TEXT("Y-Koordinate: %i\n"), data.m_mercator_y);
                TRACE(TEXT("Result Address: %s\n"), data.m_result_address);
            }
        }
        else if ( (LRESULT)wParam == RI_NOTAV )
        {
            //SearchAddress has no street found. errorcode (19)
            //get the data from the shared memory
            RI_CSearchAddress data;
            LRESULT read_suc = RI_SearchAddress_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )<BR>
                //...data was overwritten. errorcode (-2)
            else
            {
                TRACE(TEXT("X-Koordinate: %i\n"), data.m_mercator_x);
                TRACE(TEXT("Y-Koordinate: %i\n"), data.m_mercator_y);
                TRACE(TEXT("Result Address: %s\n"), data.m_result_address);
            }
        }
        else
            //SearchAddress not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_SearchAddress::foo()
{
    //*** copy data ***
    RI_CSearchAddress data;
    wcscpy(data.m_address, L"D,76131,Karlsruhe,Stumpfstrasse,1");

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SearchAddress_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_SEARCHADDRESS, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SEARCHADDRESS );
/** @ingroup DeleteAllStopOffPoints
 *  Message ID for the Request "delete all stop off points":<BR>
	RI_MESSAGE_DELETEALLSTOPOFFPOINTS

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for DeleteAllStopOffPoints:</H3>
	RI_NOERROR = Delete all Stop of Points succeeded<BR>

	<h2>Description:</h2>Deletes the complete list of stop-off-points.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) )
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_DeleteAllStopOffPoints::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI

    if ( message == RI_MESSAGE_DELETEALLSTOPOFFPOINTS )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Delete all Stop of Points succeeded. errorcode (0)
        else
            //Delete all Stop of Points not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_DeleteAllStopOffPoints::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_DELETEALLSTOPOFFPOINTS, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_DELETEALLSTOPOFFPOINTS );
/** @ingroup AddStopOffPoint_CurrentPosition
 *  Message ID for the Request "add stop off point from current position":<BR>
	RI_MESSAGE_ADDSTOPPOFFPOINT_CURRENTPOSITION

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for AddStopOffPoint_CurrentPosition:</H3>
	RI_NOERROR = Current Position succeeded<BR>
	RI_NOTADDED = Current Position not succeeded<BR>

	<h2>Description:</h2>Adds the current position to the stationlist.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_AddStopOffPoint_CurrentPosition::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI

    if ( message == RI_MESSAGE_ADDSTOPPOFFPOINT_CURRENTPOSITION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Current Position succeeded. errorcode (0)
        else
            //Current Position not succeeded. errorcode (-1)
        }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_AddStopOffPoint_CurrentPosition::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_ADDSTOPPOFFPOINT_CURRENTPOSITION, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ADDSTOPPOFFPOINT_CURRENTPOSITION );
/** @ingroup AddStopOffPoint
 *  Message ID for the Request "add stop off ponit":<BR>
	RI_MESSAGE_ADDSTOPPOFFPOINT

	<H3>Write Memory:</H3>
	RI_AddStopOffPoint_WriteData(LPARAM id, RI_CAddStopOffPoint data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_AddStopOffPoint_ReadData(LPARAM id, RI_CAddStopOffPoint data );
	<BR>
	-->
	<H3>Return Values for AddStopOffPoint:</H3>
	RI_NOERROR = AddStopOffPoint succeeded<BR>
	RI_NODESC   = AddStopOffPoint not succeeded, Description failed<BR>
	RI_NOTADDED = AddStopOffPoint not succeeded, AddStation failed<BR>
	RI_NOTLINKABLE = AddStopOffPoint not succeeded, Position not linkableNo Address found<BR><BR>

	<h2>Description:</h2>Adds a stop-off-point to the stationlist.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_AddStopOffPoint::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_ADDSTOPPOFFPOINT )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //AddStopOffPoint returned. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC )
            //AddStopOffPoint not returned, Description failed. errorcode (12)
        else if ( (LRESULT)wParam == RI_NOTLINKABLE )
            //AddStopOffPoint not returned, Position not linkable. errorcode (10)
        else if ( (LRESULT)wParam == RI_NOTADDED )
            //AddStopOffPoint not returned, AddStation failed. errorcode (11)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_AddStopOffPoint::foo()
{
    //*** copy data ***
    RI_CAddStopOffPoint data;
    data.m_mercator_x = 1019870;
    data.m_mercator_y = 6228555;
    wcscpy(data.m_Caption, L"Stuttgart");
    wcscpy(data.m_Description, L"Stuttgart");

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_AddStopOffPoint_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_ADDSTOPPOFFPOINT, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ADDSTOPPOFFPOINT );
/** @ingroup StartNavigation
 *  Message ID for the Request "start navigation":<BR>
	RI_MESSAGE_STARTNAVIGATION

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for StartNavigation:</H3>
	RI_NOERROR = Navigation succeeded<BR>
	RI_NOTADDED = Tracking Modus succeeded<BR>
	RI_NOFIX = Navigation not succeeded, no GPS Fix<BR>
	RI_NOTRUN = Navigation not succeeded, error in route calculation<BR>
	RI_USER = Navigation not succeeded, User abort Navigation<BR>
	RI_START_RETRACTED = Navigation succeeded. start was retracted<BR>
	RI_TARGET_RETRACTED = Navigation succeeded. Target was retracted<BR>
	RI_START_TARGET_RETRACTED = Navigation succeeded. Start and target was retracted<BR><BR>

	<h2>Description:</h2> Sending this message will start the navigation to a previously set stop-off-point.
	If no stop-off-point is given, the navigator will go into tracking mode.<br>
	If there's no gps fix, the command will wait for up to 60 seconds for gps.<br>
	The acknowledgement message of the command will return after the route calculation, so be prepared that the message can take a while to arrive.<br>
    The remote interface will disable user interaction before executing the command and allow user interaction after
    the command if there was no previously call to BlockUserInteraction.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) )
        return RI_NAVIGATIONNOTACTIVE;
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_StartNavigation::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_STARTNAVIGATION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Navigation succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_USER)
            //Navigation not succeeded, User abort Navigation. errorcode (20)
        else if ( (LRESULT)wParam == RI_NOTADDED)
            //Tracking Modus succeeded. errorcode (11)
        else if ( (LRESULT)wParam == RI_NOFIX)
            //Navigation not succeeded, no GPS Fix. errorcode (18)
        else if ( (LRESULT)wParam == RI_NOTRUN)
            //Navigation not succeeded, Route calculation failed. errorcode (17)
        else if ( (LRESULT)wParam == RI_START_RETRACTED )
            //Navigation succeeded. Start was retracted. errorcode (21)
        else if ( (LRESULT)wParam == RI_TARGET_RETRACTED )
            //Navigation succeeded. Target was retracted. errorcode (22)
        else if ( (LRESULT)wParam == RI_START_TARGET_RETRACTED )
            //Navigation succeeded. Start and Target was retracted.errorcode (23)
        else 
            //Navigation not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_StartNavigation::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_STARTNAVIGATION, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
\n
<H3>Flowchart:</H3>
\image html pics\RI-StartNavigation.gif
\n
*/
REGISTER_RIMESSAGE( RI_MESSAGE_STARTNAVIGATION );
/** @ingroup StartSimulation
 *  Message ID for the Request "start simulation":<BR>
	RI_MESSAGE_STARTSIMULATION

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for StartSimulation:</H3>
	RI_NOERROR = Simulation succeeded<BR>
	RI_NOTADDED = Simulation not succeeded<BR>
	RI_START_RETRACTED = Simulation succeeded. Start was retracted<BR>
	RI_TARGET_RETRACTED = Simulation succeeded. Target was retracted<BR>
	RI_START_TARGET_RETRACTED = Simulation succeeded. Start and target was retracted<BR><BR>
    
    <h2>Description:</h2> Sending this message will start a simulation.<br>
    Requires two previously added stop-off-points.<br>
    The remote interface will disable user interaction before executing the command and allow user interaction after
    the command if there was no previously call to BlockUserInteraction.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) )
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_StartSimulation::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_STARTSIMULATION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Simulation succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_NOTADDED)
            //Simulation not succeeded, invalid points. errorcode (11)
        else if ( (LRESULT)wParam == RI_START_RETRACTED )
            //Simulation succeeded. errorcode (21)
        else if ( (LRESULT)wParam == RI_TARGET_RETRACTED )
            //Simulation succeeded. errorcode (22)
        else if ( (LRESULT)wParam == RI_START_TARGET_RETRACTED )
            //Simulation succeeded. errorcode (23)
        else 
            //Simulation not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_StartSimulation::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_STARTSIMULATION, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
\n
<H3>Flowchart:</H3>
\image html pics\RI-StartSimulation.gif
\n
*/
REGISTER_RIMESSAGE( RI_MESSAGE_STARTSIMULATION );
/** @ingroup EndApplication
 *  Message ID for the Request "end application":<br>
	RI_MESSAGE_ENDAPPLICATION

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for EndApplication:</H3>
	RI_NOERROR = End Application succeeded.<BR>
	
	<h2>Description:</h2> Sending this message will end the application.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_EndApplication::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    //RI_MESSAGE_ENDAPPLICATION
    if ( message == RI_MESSAGE_ENDAPPLICATION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //End Application succeeded. errorcode (0)
        else
            //End Application not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_EndApplication::foo()
{
    //funktion end application
    LPARAM id = GetUniqueID();
    //RI_MESSAGE_ENDAPPLICATION
    if ( RI_MESSAGE( RI_MESSAGE_ENDAPPLICATION, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ENDAPPLICATION );
/** @ingroup AddStopOffPoint2FavoriteList
 *  Message ID for the Request "add stop off ponit to favoritelist":<BR>
	RI_MESSAGE_ADDSTOPOFFPOINTFAVORITE

	<H3>Write Memory:</H3>
	RI_AddStopOffPoint2FavoriteList_WriteData(LPARAM id, RI_CAddStopOffPoint2FavoriteList data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_AddStopOffPoint2FavoriteList_ReadData(LPARAM id, RI_CAddStopOffPoint2FavoriteList data );
	<BR>
	-->
	<H3>Return Values for AddStopOffPoint2FavoriteList:</H3>
	RI_NOERROR = Add Stop of Point to FavoriteList succeeded<BR>
	RI_NODESC   = Add Stop of Point to FavoriteList not succeeded, because description failed<BR>
	RI_NOTADDED = Add Stop of Point to FavoriteList not succeeded, because AddStation failed<BR><BR>

	<h2>Description:</h2>Adds a stop-off-point to the favourite list.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_AddStopOffPoint2FavoriteList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_ADDSTOPOFFPOINTFAVORITE )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Add Stop of Point to FavoriteList succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_NOTADDED )
        {
            //Add Stop of Point to FavoriteList not succeeded, because AddStation failed. errorcode (11)
        }
        else if ( (LRESULT)wParam == RI_NODESC )
        {
            //Add Stop of Point to FavoriteList not succeeded, because description failed. errorcode (12)
        }
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_AddStopOffPoint2FavoriteList::foo()
{
    //*** copy data ***
    RI_CAddStopOffPoint2FavoriteList data;
    data.m_mercator_x = 937170;
    data.m_mercator_y = 6270165;
    wcscpy(data.m_Caption, L"PTV AG Karlsruhe");
    wcscpy(data.m_Description, L"PTV AG Karlsruhe");

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_AddStopOffPoint2FavoriteList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_ADDSTOPOFFPOINTFAVORITE, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ADDSTOPOFFPOINTFAVORITE );
/** @ingroup AddStopOffPoint2PreviousDestinationList
 *  Message ID for the Request "add stop off ponit to previous destination list":<BR>
	RI_MESSAGE_ADDSTOPOFFPOINT2PDL

	<H3>Write Memory:</H3>
	RI_AddStopOffPoint2PreviousDestinationList_WriteData(LPARAM id, RI_CAddStopOffPoint2PreviousDestinationList data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_AddStopOffPoint2PreviousDestinationList_ReadData(LPARAM id, RI_CAddStopOffPoint2PreviousDestinationList data );
	<BR>
	-->
	<H3>Return Values for AddStopOffPoint2PreviousDestinationList:</H3>
	RI_NOERROR = Add Stop of Point to previous destination list succeeded<BR>
	RI_NODESC   = Add Stop of Point to previous destination list not succeeded, because description failed<BR>
	RI_NOTADDED = Add Stop of Point to previous destination list not succeeded, because AddStation failed<BR><BR>

	<h2>Description:</h2>Adds a stop-off-point to the previous destination list.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_AddStopOffPoint2PreviousDestinationList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_ADDSTOPOFFPOINT2PDL )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Add Stop of Point to PreviousDestination List succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_NOTADDED )
        {
            //Add Stop of Point to PreviousDestination List not succeeded, because AddStation failed. errorcode (11)
        }
        else if ( (LRESULT)wParam == RI_NODESC )
        {
            //Add Stop of Point to PreviousDestination List not succeeded, because description failed. errorcode (12)
        }
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_AddStopOffPoint2PreviousDestinationList::foo()
{
    //*** copy data ***
    RI_CAddStopOffPoint2PreviousDestinationList data;
    data.m_mercator_x = 937170;
    data.m_mercator_y = 6270165;
    wcscpy(data.m_Caption, L"PTV AG Karlsruhe");
    wcscpy(data.m_Description, L"PTV AG Karlsruhe");

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_AddStopOffPoint2PreviousDestinationList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_ADDSTOPOFFPOINT2PDL, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ADDSTOPOFFPOINT2PDL );
/** @ingroup GetStopOffPointList
 *  Message ID for the Request "get stop off point from stationlist":<BR>
	RI_MESSAGE_GETSTOPOFFPOINTLIST

	<H3>Write Memory:</H3>
	RI_GetStopOffPointList_WriteData(LPARAM id, RI_CGetStopOffPointList data );
	<BR>
	<H3>Read Memory:</H3>
	RI_GetStopOffPointList_ReadData(LPARAM id, RI_CGetStopOffPointList data );
	<BR>
	<H3>Return Values for AddStopOffPoint2PreviousDestinationList:</H3>
	RI_NOERROR = Get Stop of Point List succeeded<BR><BR>

	<h2>Description:</h2>Gets all stop-off-points.<br>
	If more than one station is in the list, the first received data structure will include the number
	of stations. The user will have explicitly send a new message to get the next station until all stations
	are transfered.
    <h3>Can return multiple packages</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GetStopOffPointList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_GETSTOPOFFPOINTLIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {   
            RI_CGetStopOffPointList data;
            LRESULT read_suc = RI_GetStopOffPointList_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
            //...data was overwritten. errorcode (-2)
            else
            {   
                //examine whether the current station-index is smaller than the stationcount
                if (data.m_index < data.m_listcount )
                {
                    //show data:
                    TRACE(TEXT("Caption: %s\n"),data.m_Caption);
                    TRACE(TEXT("Description: %s\n"), data.m_Description);
                    TRACE(TEXT("X-Koorinate: %i\n"), data.m_mercator_x);
                    TRACE(TEXT("Y-Koordinate: %i\n"), data.m_mercator_y);
                    TRACE(TEXT("ID: %i"),data.m_StationID);
                    //advance station counter 1
                    data.m_index = data.m_index + 1;
                    //write data
                    LRESULT sharing_ret = RI_GetStopOffPointList_WriteData( data.m_ID, data );
                    //get next station from the list
                    RI_MESSAGE( RI_MESSAGE_GETSTOPOFFPOINTLIST, GetSafeHwnd(), data.m_ID );
                }
            }

        }
        else 
            //Get Stop of Point List not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GetStopOffPointList::foo()
{
    //*** copy data ***
    RI_CGetStopOffPointList data;
    //write flag and index
    data.m_index = 0;

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_GetStopOffPointList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_GETSTOPOFFPOINTLIST, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_GETSTOPOFFPOINTLIST );
/** @ingroup ClearStopOffPoint
 *  Message ID for the Request "delete stop off point from stationlist":<BR>
	RI_MESSAGE_DELSTOPOFFPOINTFROMLIST

	<H3>Write Memory:</H3>
	RI_ClearStopOffPoint_WriteData(LPARAM id, RI_CClearStopOffPoint data );
	<BR>
	<H3>Read Memory:</H3>
	RI_ClearStopOffPoint_ReadData(LPARAM id, RI_CClearStopOffPoint data );
	<BR>
	<H3>Return Values for ClearStopOffPoint:</H3>
	RI_NOERROR = The stop off point was successfully deleted<BR>
	RI_NOTFOUND = Stop off Point was not found<BR><BR>

	<h2>Description:</h2>Deletes a station from the stationlist.
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ClearStopOffPoint::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_DELSTOPOFFPOINTFROMLIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CClearStopOffPoint data;
            LRESULT read_suc = RI_ClearStopOffPoint_ReadData( lParam, data );
            TRACE( TEXT("The stop off point with the unique ID %i"),data.m_StationID );
            TRACE( TEXT("was successfully deleted. errorcode (0)\r\n") );
        }
        else 
            //Stop off Point was not removed. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ClearStopOffPoint::foo()
{
    //*** copy data ***
    RI_CClearStopOffPoint data;
    //write UniqueId from Stop off Point
    data.m_StationID = 47;

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_ClearStopOffPoint_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_DELSTOPOFFPOINTFROMLIST, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_DELSTOPOFFPOINTFROMLIST );
/** @ingroup GetFavoriteList
 *  Message ID for the Request "Get Favoritelist":<BR>
	RI_MESSAGE_GETFAVORITELIST

	<H3>Write Memory:</H3>
	RI_GetFavoriteList_WriteData(LPARAM id, RI_CGetFavoriteList data );
	<BR>
	<H3>Read Memory:</H3>
	RI_GetFavoriteList_ReadData(LPARAM id, RI_CGetFavoriteList data );
	<BR>
	<H3>Return Values for GetFavoriteList:</H3>
	RI_NOERROR = Get Favorite List succeeded<BR><BR>

	<h2>Description:</h2>Gets the favorite list.<br>
	If more than one station is in the list, the first received data structure will include the number
	of stations. The user will have explicitly send a new message to get the next station until all stations
	are transfered.
    <h3>Can return multiple packages</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GetFavoriteList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_GETFAVORITELIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CGetFavoriteList data;
            LRESULT read_suc = RI_GetFavoriteList_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten. errorcode (-2)
            else
            {
                //examine whether station is smaller than more stationscounter
                if (data.m_index < data.m_listcount )
                {
                    TRACE(TEXT("Caption: %s"),data.m_Caption);
                    TRACE(TEXT("Description: %s"),data.m_Description);
                    TRACE(TEXT("X-Koordinate: %i"),data.m_mercator_x);
                    TRACE(TEXT("Y-Koordinate: %i"),data.m_mercator_y);
                    TRACE(TEXT("ID: %i"),data.m_StationID);
                    //advance station counter
                    data.m_index = data.m_index + 1;
                    //write data
                    LRESULT sharing_ret = RI_GetFavoriteList_WriteData( data.m_ID, data );
                    //get next station from the list
                    RI_MESSAGE( RI_MESSAGE_GETFAVORITELIST, GetSafeHwnd(), data.m_ID );
                 }
            }
        }
        else 
            //Get Stop of Point List not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GetFavoriteList::foo()
{
    //*** copy data ***
    RI_CGetFavoriteList data;
    //write flag and index
    data.m_index = 0;

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_GetFavoriteList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_GETFAVORITELIST, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_GETFAVORITELIST );
/** @ingroup ClearStopOffPointFromFavoriteList
 *  Message ID for the Request "Clear Stop Off Point From FavoriteList":<BR>
	RI_MESSAGE_DELFROMFAVORITELIST

	<H3>Write Memory:</H3>
	RI_ClearStopOffPointFromFavoriteList_WriteData(LPARAM id, RI_CClearStopOffPointFromFavoriteList data );
	<BR>
	<H3>Read Memory:</H3>
	RI_ClearStopOffPointFromFavoriteList_ReadData(LPARAM id, RI_CClearStopOffPointFromFavoriteList data );
	<BR>
	<H3>Return Values for ClearStopOffPointFromFavoriteList:</H3>
	RI_NOERROR = The stop off point was successfully deleted<BR>
	RI_NOTFOUND = The stop off point was  not found<BR><BR>

	<h2>Description:</h2>Deletes a station from the favorites list.
	<h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ClearStopOffPointFromFavoriteList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_DELFROMFAVORITELIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CClearStopOffPointFromFavoriteList data;
            LRESULT read_suc = RI_ClearStopOffPointFromFavoriteList_ReadData( lParam, data );
            TRACE( TEXT("The stop off point with the unique ID %i"),data.m_StationID );
            TRACE( TEXT("was successfully deleted. errorcode (0)\r\n") );
        }
        else if ( (LRESULT)wParam == RI_NOTFOUND )
            //Stop off Point was not removed, because no Station found. errorcode (5)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ClearStopOffPointFromFavoriteList::foo()
{
    //*** copy data ***
    RI_CClearStopOffPointFromFavoriteList data;
    //write UniqueID from Station
    data.m_StationID = 348;

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_ClearStopOffPointFromFavoriteList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_DELFROMFAVORITELIST, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_DELFROMFAVORITELIST );
/** @ingroup GetPreviousDestinationList
 *  Message ID for the Request "Get stop off point from Previous DestinationList":<BR>
	RI_MESSAGE_GETDESTINATIONLIST

	<H3>Write Memory:</H3>
	RI_GetPreviousDestinationList_WriteData(LPARAM id, RI_CGetPreviousDestinationList data );
	<BR>
	<H3>Read Memory:</H3>
	RI_GetPreviousDestinationList_ReadData(LPARAM id, RI_CGetPreviousDestinationList data );
	<BR>
	<H3>Return Values for GetPreviousDestinationList:</H3>
	RI_NOERROR = The stop off point was successfully deleted<BR><BR>

	<h2>Description:</h2>Gets the previous destination list.
	If more than one station is in the list, the first received data structure will include the number
	of stations. The user will have explicitly send a new message to get the next station until all stations
	are transfered.
    <h3>Can return multiple packages</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GetPreviousDestinationList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_GETDESTINATIONLIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CGetPreviousDestinationList data;
            LRESULT read_suc = RI_GetPreviousDestinationList_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten. errorcode (-2)
            else
            {
                //examine whether station is smaller than more stationscounter
                if (data.m_index < data.m_listcount )
                {
                    TRACE(TEXT("Caption: %s"),data.m_Caption);
                    TRACE(TEXT("Description: %s"),data.m_Description);
                    TRACE(TEXT("X-Koordinate: %i"),data.m_mercator_x);
                    TRACE(TEXT("Y-Koordinate: %i"),data.m_mercator_y);
                    TRACE(TEXT("ID: %i"),data.m_StationID);
                    //advance station counter
                    data.m_index = data.m_index + 1;
                    //write data
                    LRESULT sharing_ret = RI_GetPreviousDestinationList_WriteData( data.m_ID, data );
                    //get next station from the list
                    RI_MESSAGE( RI_MESSAGE_GETDESTINATIONLIST, GetSafeHwnd(), data.m_ID );
                }
            }
        }
        else 
            //Get Stop of Point List not succeeded. errorcode (-1)
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GetPreviousDestinationList::foo()
{
    //*** copy data ***
    RI_CGetPreviousDestinationList data;
    //write flag and index
    data.m_index = 0;

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_GetPreviousDestinationList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_GETDESTINATIONLIST, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_GETDESTINATIONLIST );
/** @ingroup ClearStopOffPointFromPreviousDestinationList
 *  Message ID for the Request "Clear Stop Off Point From Previous DestinationList":<BR>
	RI_MESSAGE_DELFROMDESTINATIONLIST

	<H3>Write Memory:</H3>
	RI_ClearStopOffPointFromPreviousDestinationList_WriteData(LPARAM id, RI_CClearStopOffPointFromPreviousDestinationList data );
	<BR>
	<H3>Read Memory:</H3>
	RI_ClearStopOffPointFromPreviousDestinationList_ReadData(LPARAM id, RI_CClearStopOffPointFromPreviousDestinationList data );
	<BR>
	<H3>Return Values for ClearStopOffPointFromPreviousDestinationList:</H3>
	RI_NOERROR = The stop off point was successfully deleted<BR>
	RI_NOTFOUND = The stop off point was not found<BR><BR>

	<h2>Description:</h2>Deletes a stop-off-point (referenced by the id of the point in the list)
	from the previous destination list.
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ClearStopOffPointFromPreviousDestinationList::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI

    if ( message == RI_MESSAGE_DELFROMDESTINATIONLIST )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CClearStopOffPointFromPreviousDestinationList data;
            LRESULT read_suc = RI_ClearStopOffPointFromPreviousDestinationList_ReadData( lParam, data );
            CString dummy;
            TRACE(TEXT("The stop off point with the unique ID %i"),data.m_StationID );
            TRACE(TEXT("was successfully deleted. errorcode (0)\r\n") );
        }
        else if ( (LRESULT)wParam == RI_NOTFOUND )
            //Stop off Point was not removed, because the Station was not found. errorcode (5)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ClearStopOffPointFromPreviousDestinationList::foo()
{
    //*** copy data ***
    RI_CClearStopOffPointFromPreviousDestinationList data;
    //write UniqueID from Station
    data.m_StationID = 256;

    //*** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_ClearStopOffPointFromPreviousDestinationList_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR )
        return;
    if ( RI_MESSAGE( RI_MESSAGE_DELFROMDESTINATIONLIST, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_DELFROMDESTINATIONLIST );
/** @ingroup ActualApplicationState
 *  Message ID for the Request "actual application state":<BR>
	RI_MESSAGE_CURRENTSTATE

<!--
	<H3>Write Memory:</H3>
	//RI_ActualApplicationState_WriteData(LPARAM id, RI_CActualApplicationState data );
	//<BR>
-->
	<H3>Read Memory:</H3>
	RI_ActualApplicationState_ReadData(LPARAM id, RI_CActualApplicationState data );
	<BR>
	<H3>Return Values for ActualApplicationState:</H3>
	RI_NOERROR = Actual state of the application succeeded<BR><BR>

	<h2>Description:</h2>Returns the current application state as a string. The state represents the current dialog which is shown and can be one of the following strings:<br>
	"Intro About ", "MainMenu ", "StationList ", "StationListOptions ", "Tour ", "Navigation ", "Route ", "RouteList ", "DefineDestination ", "DefineDestinationTown ", "DefineDestinationStreet ", "DefineDestinationHouseNr ", "DefineDestinationMap ", "DefineDestinationFavourites ", "DefineDestinationPrevDest ", "TmcMsgs ", "GpsWait ", "GpsInfo ", "NavInfo ", "PreDestinationList ", "Settings ", "SettingsAvoidsect ", "SettingsBaud ", "SettingsBrightness ", "SettingsBrightness ", "SettingsCopyright ", "SettingsDatalink ", "SettingsFavouritesAdminister ", "SettingsFavouritesDetails ", "SettingsFavouritesRename ", "SettingsGeneral ", "SettingsLanguage ", "SettingsMap ", "SettingsPOI ", "SettingsPort ", "SettingsProfile ", "SettingsProfileOverview ", "SettingsStreetPreference ", "SettingsTechnicalLKWData ", "SettingsSaveProfile ", "SettingsRouteoptimisation ", "SettingsSound ", "SettingsTMC ", "TmcListOptions ", "SettingsVehicle ", "SettingsTimeZone ", "SettingsInfo ", "TextBrowser ", "unkown ".<br>
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ActualApplicationState::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_CURRENTSTATE )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CActualApplicationState data;
            LRESULT read_suc = RI_ActualApplicationState_ReadData( lParam, data );
            //Actual state of the application succeeded. errorcode (0)
            TRACE( TEXT("the actual state of the application is:  %s"),data.m_Description );
        }
        else
            //Actual state of the application not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ActualApplicationState::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_CURRENTSTATE, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_CURRENTSTATE );
/** @ingroup StartGPSRecord
 *  Message ID for the Request "start GPS Record":<BR>
	RI_MESSAGE_STARTGPSRECORD

	<H3>Write Memory:</H3>
	RI_StartGPSRecord_WriteData(LPARAM id, RI_CStartGPSRecord data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_StartGPSRecord_ReadData(LPARAM id, RI_CStartGPSRecord data );
	<BR>
	-->
	<H3>Return Values for StartGPSRecord:</H3>
	RI_NOERROR = Start recording of GPS positions succeeded<BR>
	RI_TOBIG = GPS recording file reached maximum file size<BR>
	RI_ALREADY = Start recording of GPS positions is already running<BR>
	RI_NOTWRITE = Error while opening GPS recording file<BR><BR>

	<h2>Description: </h2>Records the GPS-coordinates in a separate file (see data structure for
	more details).
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_StartGPSRecord::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_STARTGPSRECORD )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Start recording of GPS positions succeeded. errorcode(0)
        else if ( (LRESULT)wParam == RI_ALREADY )
            //Start recording of GPS positions is already running. errorcode(13)
        else if ( (LRESULT)wParam == RI_NOTWRITE )
            //Error while opening GPS recording file. errorcode(15)
        else if ( (LRESULT)wParam == RI_TOBIG )
            //GPS recording file reached maximum file size. errorcode(14)
        else 
            //Start recording of GPS positions not succeeded. errorcode(-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_StartGPSRecord::foo()
{
    //define struct
    RI_CStartGPSRecord data;
    // *** copy data ***
    wcscpy (data.m_FileName, L"gpsrecord.out");
    data.m_MaxFileSize = 1048576;
    data.m_Interval = 10;
    data.m_OutputFormat = 1;
    // *** write data to shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_StartGPSRecord_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_STARTGPSRECORD, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_STARTGPSRECORD );
/** @ingroup SystemInfo
 *  Message ID for the Request "system information":<BR>
	RI_MESSAGE_SYSTEMINFO

	<!--
	<H3>Write Memory:</H3>
	RI_SystemInfo_WriteData(LPARAM id, RI_CSystemInfo data );
	<BR>
	-->
	<H3>Read Memory:</H3>
	RI_SystemInfo_ReadData(LPARAM id, RI_CSystemInfo data );
	<BR>
	<H3>Return Values for SystemInfo:</H3>
	RI_NOERROR = System info succeeded<BR><BR>

	<h2>Description:</h2>Returns system information like the current language, available languages,
	the current map, available maps and so on (see data structure for more details).
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_SystemInfo::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SYSTEMINFO )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //read data from Shared Memory
            RI_CSystemInfo data;
            LRESULT read_suc = RI_SystemInfo_ReadData( lParam, data );
            //System info succeeded. errorcode (0)
            TRACE( TEXT("Programm Version:  %s"),data.m_Version );
            TRACE( TEXT("Programm Build:  %s"),data.m_Build );
            TRACE( TEXT("Current Map Name:  %s"),data.m_CurrentMapName);
            TRACE( TEXT("Current Map Date:  %s"),data.m_CurrentMapDate );
            TRACE( TEXT("Current Map Version:  %s"),data.m_CurrentMapVersion );
            for (int i=0; i<data.m_MapCount; i++)
            {
                TRACE(TEXT("Available Maps: %"s),data.m_AvailableMaps[i]);
            }
            TRACE( TEXT("Current Language:  %s"),data.m_CurrentLanguage );
            TRACE( TEXT("Available Language:  %s"),data.m_AvailableLanguages );
        }
        else
            //System info not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_SystemInfo::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_SYSTEMINFO, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SYSTEMINFO );
/** @ingroup SwitchMap
 *  Message ID for the Request "SwitchMap":<BR>
	RI_MESSAGE_SWITCHMAP

	<H3>Write Memory:</H3>
	RI_SwitchMap_WriteData(LPARAM id, RI_CSwitchMap data );
	<BR>
	<H3>Read Memory:</H3>
	RI_SwitchMap_ReadData(LPARAM id, RI_CSwitchMap data );
	<BR>
	<H3>Return Values for SwitchMap:</H3>
	RI_NOERROR = SwitchMap succeeded<BR>
	RI_NOTADDED = Switch Map not succeeded, because no map found<BR><BR>

	<h2>Description:</h2>Switches the map to the given one. If the map doesn't exist, returns an error.
	Also returns the available maps as a string list.
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_SwitchMap::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SWITCHMAP )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CSwitchMap data;
            LRESULT read_suc = RI_SwitchMap_ReadData( lParam, data );
            //Switch Map succeeded. errorcode (0)
            TRACE( TEXT("Map Name:  %s"),data.m_MapName );
            TRACE( TEXT("Current Map Name:  %s"),data.m_CurrentMapName );
            TRACE( TEXT("Current Map Date:  %s"),data.m_Currentmapdate );
            TRACE( TEXT("Current Map Version:  %s"),data.m_CurrentMapversion );
            for (int i=0; i<data.m_MapCount; i++)
            {
                TRACE(TEXT("Available Maps: %s"),data.m_AvailableMaps[i]);
            }
        }
        else if ( (LRESULT)wParam == RI_NOTADDED )
            //Switch Map not succeeded. errorcode (-1)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_SwitchMap::foo()
{
    //define struct
    RI_CSwitchMap data;
    // *** copy data ***
    wcscpy (data.m_MapName, L"deu_08Q2");
    // *** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SwitchMap_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_SWITCHMAP, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SWITCHMAP );
/** @ingroup ChangeApplicationOptions
 *  Message ID for the Request "ChangeApplicationOptions":<BR>
	RI_MESSAGE_CHANGEOPTIONS

	<H3>Write Memory:</H3>
	RI_ChangeApplicationOptions_WriteData(LPARAM id, RI_CChangeApplicationOptions data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_ChangeApplicationOptions_ReadData(LPARAM id, RI_CChangeApplicationOptions data );
	<BR>
	-->
	<H3>Return Values for ChangeApplicationOptions:</H3>
	RI_NOERROR = Change application options succeeded<BR><BR>

	<h2>Description:</h2>Set application options like using ferries or motorways, volume, avoid
	u-turns and so on (see data structure for more details).
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ChangeApplicationOptions::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_CHANGEOPTIONS )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Change application options succeeded. errorcode(0)
        else
            //Change application options not succeeded. errorcode(-1)
    }	
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ChangeApplicationOptions::foo()
{
    //define struct
    RI_CChangeApplicationOptions data;
    // *** copy data ***
    data.m_ShowStreetNames = 1;
    data.m_ShowCompass = 1;
    data.m_ShowScale = 1;
    data.m_MapFacingNorth = 0;
    data.m_InfoLeftSide = 0;
    data.m_InfoRightSide = 0;
    data.m_DependentOnSpeed = 0;
    data.m_Volume = 80;
    data.m_AvoidUTurns = 1;
    data.m_MapInItinerary = 0;
    data.m_Vehicle = 6;
    data.m_Optimisation = 2;
    data.m_Motorway = 1;
    data.m_Ferries = 1;
    // *** write data in shared memory ***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_ChangeApplicationOptions_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_CHANGEOPTIONS, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_CHANGEOPTIONS );
/** @ingroup StopGPSRecord
 *  Message ID for the Request "stop GPS Record":<BR>
	RI_MESSAGE_STOPGPSRECORD

	<!--
	<H3>Write Memory:</H3>
	RI_StopGPSRecord_WriteData(LPARAM id, RI_CStopGPSRecord data );
	<BR>
	-->
	<H3>Read Memory:</H3>
	RI_StopGPSRecord_ReadData(LPARAM id, RI_CStopGPSRecord data );
	<BR>
	<H3>Return Values for StopGPSRecord:</H3>
	RI_STOPGPS = Stop GPS recording succeeded<BR>
	RI_NOTRUN = GPS recording was not running<BR><BR>

	<h2>Description:</h2>Stops the GPS record.
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_StopGPSRecord::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_STOPGPSRECORD )
    {
        if ( (LRESULT)wParam == RI_STOPGPS )
        {
            RI_CStopGPSRecord data;
            //read data from shared memory
            LRESULT read_suc = RI_StopGPSRecord_ReadData( lParam, data );
            //Stop GPS recording succeeded. errorcode(16)
            TRACE(TEXT("Number Entries:  %i"),data.m_NumberEntries );
            TRACE(TEXT("File Size:  %i"),data.m_FileSize );
        }
        else if ( (LRESULT)wParam == RI_NOTRUN )
            //GPS recording was not running. errorcode(17)
    }	
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_StopGPSRecord::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_STOPGPSRECORD, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_STOPGPSRECORD );
/** @ingroup ShowMessage
 *  Message ID for the Request "show message":<BR>
	RI_MESSAGE_SHOWMESSAGE

	<H3>Write Memory:</H3>
	RI_ShowMessage_WriteData(LPARAM id, RI_CShowMessage data );
	<BR>
	<H3>Read Memory:</H3>
	RI_ShowMessage_ReadData(LPARAM id, RI_CShowMessage data );
	<BR>
	<H3>Return Values for ShowMessage:</H3>
	RI_NOERROR = Message was shown<BR>
	RI_NODESC = Message not shown, because no message was defined<BR><BR>

	<h2>Description:</h2>Shows a Messagebox with Ok, Ok/Cancel or Yes/No-Button(s). Returns the button pressed from the user (see data-struct for details).
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ShowMessage::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SHOWMESSAGE )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //Message was shown. errorcode (0)
            RI_CShowMessage data;
            //read data from shared memory
            LRESULT read_suc = RI_ShowMessage_ReadData( lParam, data );
            TRACE(TEXT("Feedback:  %i"),data.m_Feedback );
        }
        else if ( (LRESULT)wParam == RI_NODESC )
            //Message was not shown, because no Message defined. errorcode (12)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ShowMessage::foo()
{
    // *** copy data to the struct ***
    RI_CShowMessage data;
    wcscpy (data.m_MessageCaption, L"Information");
    wcscpy (data.m_Message, L"You have reached the maximum driving time!");
    data.m_MessageButton = 1;
    // *** write data in the Shared memory***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_ShowMessage_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_SHOWMESSAGE, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SHOWMESSAGE );
/** @ingroup RoutePlanning
 *  Message ID for the Request "route planning":<BR>
	RI_MESSAGE_ROUTEPLANNING

	<H3>Write Memory:</H3>
	RI_RoutePlanning_WriteData(LPARAM id, RI_CRoutePlanning data );
	<BR>
	<H3>Read Memory:</H3>
	RI_RoutePlanning_ReadData(LPARAM id, RI_CRoutePlanning data );
	<BR>
	
	<H3>Return Values for RoutePlanning:</H3>
	RI_NOERROR = Route planning started<BR>
	RI_NOTADDED = Route planning not started, because RoutingStart or RoutingEnd failed<BR>
	RI_NOTLINKABLE = Route planning not started, because route calculation failed<BR><BR>

	<h2>Description:</h2>Calculate the route with the previously sent start- and endstations.
	The first read packet includes the number of maneuvers of the calculated route.
	The user have to explicitly send a new message to get the next maneuver description until
	all maneuvers are transfered.
    The remote interface will disable user interaction before executing the command and allow user interaction after
    the command if there was no previously call to BlockUserInteraction.
    <h3>Can return multiple packages.</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_RoutePlanning::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_ROUTEPLANNING )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CRoutePlanning data;
            LRESULT read_suc = RI_RoutePlanning_ReadData( lParam, data );
            if (data.m_flag != true)
            {
                //Route planning started. errorcode (0)
                TRACE(TEXT("Retracted: %i"), data.m_Retracted);
                TRACE(TEXT("Number of Manoeuvres: %i"),data.m_ListCount);
                TRACE(TEXT("Total Road: %i"),data.m_TotalTime);
                TRACE(TEXT("Total Duration: %i"),data.m_TotalDistance);
            }
            if (data.m_index <= data.m_ListCount )
            {
                TRACE(TEXT("Station: %i"),data.m_index);
                TRACE(TEXT("Manoeuvre Text: %s"),data.m_ManeuverText);
                TRACE(TEXT("Road: %i"),data.m_Road);
                TRACE(TEXT("Direction: %i"),data.m_Direction);
                TRACE(TEXT("Coordinates: %i,%i"),data.m_mercator_x, data.m_mercator_y);
                //advance station counter 1
                data.m_index = data.m_index + 1;
                //flag suppresses succedd the message for the next station
                data.m_flag = true;
                //write data
                LRESULT sharing_ret = RI_RoutePlanning_WriteData( data.m_ID, data );
                //get next station from the list
                RI_MESSAGE( RI_MESSAGE_ROUTEPLANNING, GetSafeHwnd(), data.m_ID );
            }
        }
        else if ( (LRESULT)wParam == RI_NOTADDED )
            //Route planning not started, because RoutingStart or RoutingEnd failed. errorcode (11)
        else if ( (LRESULT)wParam == RI_NOTLINKABLE )
            //Route planning not started, because route calculation failed. errorcode (10)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_RoutePlanning::foo()
{
    //define struct
    RI_CRoutePlanning data;
    data.m_index = 1;
    data.m_flag = false;
    // *** write data in the Shared memory***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_RoutePlanning_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_ROUTEPLANNING, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
\n
<H3>Flowchart:</H3>
\image html pics\RI-StartRouteplanning.gif
\n
*/
REGISTER_RIMESSAGE( RI_MESSAGE_ROUTEPLANNING );
/** @ingroup InverseGeoCoding
 *  Message ID for the Request "inverse geocoding":<BR>
	RI_MESSAGE_INVERSEGEOCODING

	<H3>Write Memory:</H3>
	RI_InverseGeoCoding_WriteData(LPARAM id, RI_CInverseGeoCoding data );
	<BR>
	<H3>Read Memory:</H3>
	RI_InverseGeoCoding_ReadData(LPARAM id, RI_CInverseGeoCoding data );
	<BR>
	<H3>Return Values for InverseGeoCoding:</H3>
	RI_NOERROR = Inverse GeoCoding succeeded.<BR>
	RI_NOTADDED = Inverse GeoCoding not succeeded, because address not found in map.<BR><BR>

	<h2>Description:</h2>Returns an address string for a given mercator pos.<br>
	The output is given as a formatted string like the following:<br>
	D,76131,Karlsruhe,Tullastrasse,1-11|2-20<br>
	The housenumber will be given as ranges separated by the pipe '|' symbol. A range of 0-0 means no
	housenumber found for this side of the segment. The different ranges will always be sorted from 
	lower to upper.

    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    // send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_InverseGeoCoding::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_INVERSEGEOCODING )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            RI_CInverseGeoCoding data;
            //Read Data from shared memory
            LRESULT read_suc = RI_InverseGeoCoding_ReadData( lParam, data );
            //Inverse GeoCoding succeeded. errorcode (0)
            TRACE(TEXT("Result Address: %s" ),data.m_address); 
            AppendStatusText( TEXT("\r\n") );
        }
        else if ( (LRESULT)wParam == RI_NOTADDED )
            //Inverse GeoCoding not succeeded, because Address not found in map. errorcode (11)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_InverseGeoCoding::foo()
{
    //define struct
    RI_CInverseGeoCoding data;
    data.m_mercator_x = 937170;
    data.m_mercator_y = 6270165;
    // *** write data in the Shared memory***
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_InverseGeoCoding_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_INVERSEGEOCODING, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_INVERSEGEOCODING );
/** @ingroup GPSInfo
 *  Message ID for the Request "GPS info":<BR>
	RI_MESSAGE_GPSINFO

	<!--
	<H3>Write Memory:</H3>
	RI_GPSInfo_WriteData(LPARAM id, RI_CGPSInfo data );
	<BR>
	-->
	<H3>Read Memory:</H3>
	RI_GPSInfo_ReadData(LPARAM id, RI_CGPSInfo data );
	<BR>
	<H3>Return Values for InverseGeoCoding:</H3>
	RI_NOERROR = GPS Signal ok<BR>
	RI_NOFIX = No GPS fix<BR>
	RI_NOTAV = No GPS Device available<BR><BR>

	<h2>Description:</h2>Returns GPS-Informations like position and velocity (see data structure for
	more details).
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GPSInfo::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_GPSINFO )
    {
        if ( (LRESULT)wParam == RI_NOFIX )
            //No Fix. errorcode (18)
        else if ( (LRESULT)wParam == RI_NOTAV)
            //No GPS Device available. errorcode (19)
        else if ( (LRESULT)wParam == RI_NOERROR)
        {
            //read data from Shared Memory
            RI_CGPSInfo data;
            LRESULT read_suc = RI_GPSInfo_ReadData( lParam, data );
            //GPS Signal ok.errorcode (0)
            TRACE(TEXT("Geodec Pos: %i,%i"), data.m_geodec_x, data.m_geodec_y);
            TRACE(TEXT("Mercator Pos: %i,%i"), data.m_mercator_x, data.m_mercator_y);
            TRACE(TEXT("Longitude: %i"), data.m_lon);
            TRACE(TEXT("Latitude: %i"), data.m_lat);
            TRACE(TEXT("Course: %i"), data.m_course);
            TRACE(TEXT("Speed: %i"), data.m_velocity);
            TRACE(TEXT("Satellites: %i"), data.m_noSatsUsed);
            TRACE(TEXT("Altitude: %i"), data.m_Altitude);
            TRACE(TEXT("Satellite Date: %s"), data.m_date);
            TRACE(TEXT("Satellite Time: %s"), data.m_time);
            TRACE(TEXT("Last Valid Pos: %i"), data.m_lastdata);
            //read data, when navigation is running (data.flag == true)
            if (data.flag == true)
            {
                TRACE(TEXT("Current Street: %s"), data.m_street);
                TRACE(TEXT("Target: %s"), data.m_target);
                TRACE(TEXT("Roadmatched Mercator Pos: %s"), data.n_mercator);
                TRACE(TEXT("Roadmatched Geodec. Pos: %i,%i"), data.n_geodec_x, data.n_geodec_y);
                TRACE(TEXT("Distance to Target: %i"), data.m_disttotarget);
                TRACE(TEXT("Time to Target: %i"), data.m_timetotarget);
                TRACE(TEXT("SpeedLimit: %i"), data.m_SpeedLimit);
            }
        }	
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GPSInfo::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_GPSINFO, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_GPSINFO );
/** @ingroup GPSLog
 *  Message ID for the Request "transfer NMEA Data over RI":<BR>
	RI_MESSAGE_GPSLOG

	<H3>Write Memory:</H3>
	RI_GPSLog_WriteData(LPARAM id, RI_CGPSLog data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_GPSLog_ReadData(LPARAM id, RI_CGPSLog data );
	<BR>
	-->
	<H3>Return Values for GPSLog:</H3>
	RI_NOERROR = NMEA transfer succeeded<BR>
	RI_NOTRUN = GPS transfer is stopped<BR><BR>

	<h2>Description:</h2>Sends NMEA-Data over RI to the navigation software.<br>
    The data sentences "$GPGGA", "$GPRMC" and "$GPVTG" are mandatory.<br>
    Be sure that the date in the sentences is correct, because it's needed by some modules. 
    The setences has to be complete, substrings are not supported.
    Also the NMEA strings must have a correct checksum, overwise they will be discarded.<br>
    Don't use this method together with a GPS-Device, the signals would be mixed!
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GPSLog::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI

    if ( message == RI_MESSAGE_GPSLOG )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //NMEA transfer succeeded. errorcode(0)
        else if ( (LRESULT)wParam == RI_NOTRUN )
            //GPS transfer is stopped. errorcode(17)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE )
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }	
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GPSLog::foo()
{
    // *** define struct ***
    RI_CGPSLog data;
    // *** copy data ***
    wcscpy (data.m_Nmea, 
        L"$GPGGA,171710.350,4900.5533,N,00826.3949,E,1,05,3.5,168.0,M,,,,0000*02\r\n
        $GPGLL,4900.5533,N,00826.3949,E,171710.350,A*35\r\n
        $GPRMC,171710.350,A,4900.5533,N,00826.3949,E,0.0,175.8,261103,,*0E\r\n
        $GPVTG,175.8,T,,M,0.0,N,0.0,K*6B\r\n
        $GPGGA,171711.350,4900.5533,N,00826.3950,E,1,05,3.5,168.5,M,,,,0000*0E\r\n");

    //write data to shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_GPSLog_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_GPSLOG, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_GPSLOG );
/** @ingroup RefreshMap
 *  Message ID for the Request "RefreshMap":<BR>
	RI_MESSAGE_REFRESHMAP

	<H3>Write Memory:</H3>
	RI_RefreshMap_WriteData(LPARAM id, RI_CRefreshMap data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_RefreshMap_ReadData(LPARAM id, RI_CRefreshMap data );
	<BR>
	-->
	<H3>Return Values for RefreshMap:</H3>
	RI_NOERROR = RefreshMap is enabled<BR>
	RI_NODESC = RefreshMap is disabled<BR><BR>

	<h2>Description:</h2>Disables or enables the update of the map (only in simulation, tracking and
	navigation mode).
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_RefreshMap::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_REFRESHMAP )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //RefreshMap is enabled. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC )
            //RefreshMap is disabled. errorcode (12)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE )
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_RefreshMap::foo()
{
    // *** define struct ***
    RI_CRefreshMap data;
    // *** copy data ***
    data.m_refresh = true;
    //write data to shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_RefreshMap_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_REFRESHMAP, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_REFRESHMAP );
/** @ingroup NaviControls
 *  Message ID for the Request "NaviControls":<BR>
	RI_MESSAGE_NAVICONTROLS

	<H3>Write Memory:</H3>
	RI_NaviControls_WriteData(LPARAM id, RI_CNaviControls data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_NaviControls_ReadData(LPARAM id, RI_CNaviControls data );
	<BR>
	-->
	<H3>Return Values for NaviControls:</H3>
	RI_NOERROR = transfer to Control.<BR>
	RI_NODESC = Navigator is not in Navigation Mode.<BR><BR>

	<h2>Description:</h2>Zoom in, out or skip station in navigation, simulation or tracking mode.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_NaviControls::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_NAVICONTROLS )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //transfer to Control. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC )
            //Navigator is not in Navigation Mode. errorcode (12)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE )
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_NaviControls::foo()
{
    // *** define struct ***
    RI_CNaviControls data;
    // *** copy data ***
    data.m_value = 2;
    //m_value = 0 = Zoomout
    //m_value = 1 = Zoomin
    //m_value = 2 = SkipStation
    //write data to shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_NaviControls_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_NAVICONTROLS, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_NAVICONTROLS );
/** @ingroup SetMapCenter
 *  Message ID for the Request "SetMapCenter":<BR>
	RI_MESSAGE_SETMAPCENTER

	<H3>Write Memory:</H3>
	RI_SetMapCenter_WriteData(LPARAM id, RI_CSetMapCenter data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_SetMapCenter_ReadData(LPARAM id, RI_CSetMapCenter data );
	<BR>
	-->
	<H3>Return Values for SetMapCenter:</H3>
	RI_NOERROR = change MapRect success<BR>
	RI_NODESC = AutoPanning is deactivated.<BR><BR>

	<h2>Description:</h2>Sets the map center to the given position if m_panning is true.
	There is no automatic switch back to the current gps position on the map, so be sure to call this message again with m_panning = false to center the map to the current position !<br>
    Possible coordinate formats are mercator and geodecimal (see also the data-struct description).
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_SetMapCenter::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SETMAPCENTER )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //change MapRect success. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC )
            //AutoPanning is deactivated. errorcode (12)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE )
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_SetMapCenter::foo()
{
    // *** define struct ***
    RI_CSetMapCenter data;
    // *** copy data ***
    data.m_panning = true;
    // set the coordinate format to mercator
    data.m_flag = true;
    // set the map center to the following mercator coordinates
    wcscpy(data.m_PosX, L"937170");
    wcscpy(data.m_PosY, L"6270165");
    //write data to shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SetMapCenter_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_SETMAPCENTER, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SETMAPCENTER );
/** @ingroup VisibleMapRect
 *  Message ID for the Request "VisibleMapRect":<BR>
	RI_MESSAGE_VISIBLEMAPRECT

	<H3>Write Memory:</H3>
	RI_VisibleMapRect_WriteData(LPARAM id, RI_CVisibleMapRect data );
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_VisibleMapRect_ReadData(LPARAM id, RI_CVisibleMapRect data );
	<BR>
	-->
	<H3>Return Values for VisibleMapRect:</H3>
	RI_NOERROR = change MapRect success<BR>
	RI_NODESC = AutoPanning is deactivated.<BR><BR>

	<h2>Description:</h2>Sets the visible map section to the given rect if m_panning is true.
	There is no automatic switch back to the current gps position on the map, so be sure to call this message again with m_panning = false to center the map to the current position !<br>
    Possible coordinate formats are mercator and geodecimal (see also the data-struct description).
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_VisibleMapRect::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_VISIBLEMAPRECT )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //change MapRect success. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC )
            //AutoPanning is deactivated. errorcode (12)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE )
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_VisibleMapRect::foo()
{
    // *** define struct ***
    RI_CVisibleMapRect data;
    // *** copy data ***
    data.m_panning = true;
    // set the input format to mercator
    data.m_flag = true;
    wcscpy(data.m_map_bottom, L"937170");
    wcscpy(data.m_map_left, L"6270165");
    wcscpy(data.m_map_right, L"6270165");
    wcscpy(data.m_map_top, L"937170");
    //write data to shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_VisibleMapRect_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_VISIBLEMAPRECT, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_VISIBLEMAPRECT );
/** @ingroup DataUpdate
 *  Message ID for the Request "DataUpdate":<BR>
	RI_MESSAGE_DATAUPDATE

	<H3>Write Memory:</H3>
	This function has no writer.
	<BR>
	<H3>Read Memory:</H3>
	This function has no reader.
	<BR>
	<H3>Return Values for DataUpdate:</H3>
	RI_NOERROR = Route recalculation succeeded<BR>
	RI_NOTRUN = Route recalculation does not succeed because the route calculation is running<BR><BR>

	<h2>Description:</h2>Recalculates the route if application is in navigation mode.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_DataUpdate::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_DATAUPDATE )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Route recalculation succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_NOTRUN )
        {
            //Route recalculation didn't succeed, because the route calculation is running. errorcode (17)
            //please try again...
            LPARAM id = GetUniqueID();
            RI_MESSAGE( RI_MESSAGE_DATAUPDATE, GetSafeHwnd(), id );
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE )
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_DataUpdate::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_DATAUPDATE, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_DATAUPDATE);	
/** @ingroup RiDump
 *  Message ID for the Request "RiDump":<BR>
	RI_MESSAGE_RIDUMP

	<H3>Write Memory:</H3>
	RI_RiDump_WriteData(LPARAM id, RI_CRiDump data);
	<BR>
	<H3>Read Memory:</H3>
	RI_RiDump_ReadData(LPARAM id, RI_CRiDump data);
	<BR>
	<H3>Return Values for RiDump:</H3>
	RI_NOERROR = Dump is activated<BR>
	RI_NODESC = Dump is deactivated<BR><BR>

	<h2>Description:</h2>Dumps all RI-Interface transfer to a file in the remote directory.
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_RiDump::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_RIDUMP )
    {
        if ( (LRESULT)wParam == RI_NOERROR)
            //Dump is activated. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC)
            //Dump is deactivated. errorcode (12)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_RiDump::foo()
{
    //define struct
    RI_CRiDump data;
    //copy data
    data.m_dump = true;
    //write data in shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_RiDump_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_RIDUMP, GetSafeHwnd(), id ) == RI_NOERROR ) 
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_RIDUMP);	
/** @ingroup SetProfile
 *  Message ID for the Request "RiSetProfile":<BR>
	RI_MESSAGE_SETPROFILE

	<H3>Write Memory:</H3>
	RI_SetProfile_WriteData(LPARAM id, RI_CSetProfile data);
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_SetProfile_ReadData(LPARAM id, RI_CSetProfile data);
	<BR>
	-->
	<H3>Return Values for SetProfile:</H3>
	RI_NOERROR = change Profile success<BR>
	RI_NOTADDED = Profile cannot change<BR><BR>

	<h2>Description:</h2>Activates a vehicle-profile (profile has to exist in the LKWProfile-directory).

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_SetProfile::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SETPROFILE )
    {
        if ( (LRESULT)wParam == RI_NOERROR)
            //change Profile success. errorcode (0)
        else if ( (LRESULT)wParam == RI_NODESC)
            //Profile cannot change. errorcode (12)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_SetProfile::foo()
{
    //define struct
    RI_CSetProfile data;
    //copy data
    wcscpy(data.m_ProfileName, L"PKW");
    //write data in shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SetProfile_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_SETPROFILE, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SETPROFILE);
/** @ingroup GetProfile
 *  Message ID for the Request "GetProfile":<BR>
	RI_MESSAGE_GETPROFILE

	<!--
	<H3>Write Memory:</H3>
	RI_GetProfile_WriteData(LPARAM id, RI_CGetProfile data);
	<BR>
	-->
	<H3>Read Memory:</H3>
	RI_GetProfile_ReadData(LPARAM id, RI_CGetProfile data);
	<BR>
	<H3>Return Values for GetProfile:</H3>
	RI_NOERROR = GetProfile succeeded<BR><BR>

	<h2>Description:</h2>Get the current used vehicle-profile name.
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GetProfile::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_GETPROFILE )
    {
        if ( (LRESULT)wParam == RI_NOERROR)
        {
            //get Profile success. errorcode (0)
            RI_CGetProfile data;
            //read from shared memory
            LRESULT read_suc = RI_GetProfile_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten.
            else
                TRACE(TEXT("VehicleProfile: %s\r\n"),data.m_ProfileName);
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GetProfile::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_GETPROFILE, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_GETPROFILE);
/** @ingroup SetDayNight
 *  Message ID for the Request "SetDayNight":<BR>
	RI_MESSAGE_SETDAYNIGHT

	<H3>Write Memory:</H3>
	RI_SetDayNight_WriteData(LPARAM id, RI_CSetDayNight data);
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_SetDayNight_ReadData(LPARAM id, RI_CSetDayNight data);
	<BR>
	-->
	<H3>Return Values for SetDayNight:</H3>
	RI_NOERROR = switch Day/Night succeeded<BR><BR>

	<h2>Description:</h2>Switch navigator to Day/Night mode.
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_SetDayNight::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SETDAYNIGHT )
    {
        if ( (LRESULT)wParam == RI_NOERROR)
            //switch DayNight success. errorcode (0)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_SetDayNight::foo()
{
    //define struct
    RI_CSetDayNight data;
    //copy data
    data.m_bDayNight = true;
    //write data in shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SetDayNight_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_SETDAYNIGHT, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SETDAYNIGHT);
/** @ingroup GetDayNight
 *  Message ID for the Request "GetDayNight":<BR>
	RI_MESSAGE_GETDAYNIGHT

	<!--
	<H3>Write Memory:</H3>
	RI_GetDayNight_WriteData(LPARAM id, RI_CGetDayNight data);
	<BR>
	-->
	<H3>Read Memory:</H3>
	RI_GetDayNight_ReadData(LPARAM id, RI_CGetDayNight data);
	<BR>
	<H3>Return Values for GetDayNight:</H3>
	RI_NOERROR = read DayNight Mode success<BR><BR>

	<h2>Description:</h2>Gets the actual Day/Night mode (true if day).
    <h3>Returns a single package</h3>
	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_GetDayNight::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_GETDAYNIGHT )
    {
        if ( (LRESULT)wParam == RI_NOERROR)
        {
            //read DayNight Mode success. errorcode (0)
            // get the data from the shared memory
            RI_CGetDayNight data;
            LRESULT read_suc = RI_GetDayNight_ReadData( lParam, data );
            if ( read_suc != RI_NOERROR )
                //...data was overwritten. errorcode (-2)
            else
            {
                if (data.m_bDayNight)
                    TRACE(TEXT("Day Mode is set"));
                else
                    TRACE(TEXT("Night Mode is set"));
            }
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_GetDayNight::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_GETDAYNIGHT, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_GETDAYNIGHT);
/** @ingroup StartNavigationOverBCR
 *  Message ID for the Request "StartNavigationOverBCR":<BR>
	RI_MESSAGE_STARTNAVIGATIONOVERBCR

	<H3>Write Memory:</H3>
	RI_StartNavigationOverBCR_WriteData(LPARAM id, RI_CStartNavigationOverBCR data);
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_StartNavigationOverBCR_ReadData(LPARAM id, RI_CStartNavigationOverBCR data);
	<BR>
	-->
	<H3>Return Values for StartNavigationOverBCR:</H3>
	RI_NOERROR = Navigation succeeded<BR>
	RI_ALREADY = Navigation not succeeded, Navigation is already running<BR>
	RI_NOTFOUND = Navigation not succeeded, no BCR File found<BR>
	RI_NOFIX = Navigation not succeeded, no GPS Fix<BR>
	RI_NOTAV = Navigation not succeeded, no GPS avalable<BR>
	RI_START_RETRACTED = Navigation succeeded. Start retracted<BR>
	RI_TARGET_RETRACTED = Navigation succeeded. Target retracted<BR>
	RI_START_TARGET_RETRACTED = Navigation succeeded. Start and target retracted<BR><BR>

	<h2>Description:</h2>Calculates a route from an existent BCR-File and starts the navigation.<br> The file must be present in the bcr-directory of the navigation software and the BCR filename has to be given without path extensions, e.g. don't use "/truck/bcr/foo.bcr" but "foo.bcr".<br>
	If there's no gps fix, the command will wait for up to 60 seconds for gps.<br>
	The acknowledgement message of the command will return after the route calculation, so be prepared that the message can take a while to arrive.<br>
    The remote interface will disable user interaction before executing the command and allow user interaction after
    the command if there was no previously call to BlockUserInteraction.

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_StartNavigationOverBCR::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_STARTNAVIGATIONOVERBCR )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
            //Navigation succeeded. errorcode (0)
        else if ( (LRESULT)wParam == RI_NOFIX)
            //Navigation not succeeded, no GPS Fix. errorcode (18)
        else if ( (LRESULT)wParam == RI_NOTAV)
            //Navigation not succeeded, no GPS avalable. errorcode (19)
        else if ( (LRESULT)wParam == RI_NOTFOUND)
            //Navigation not succeeded, no BCR File found. errorcode (5)
        else if ( (LRESULT)wParam == RI_ALREADY)
            //Navigation not succeeded, Navigation is already running. errorcode (17)
        else if ( (LRESULT)wParam == RI_START_RETRACTED )
            //Navigation succeeded. errorcode (21)
        else if ( (LRESULT)wParam == RI_TARGET_RETRACTED )
            //Navigation succeeded. errorcode (22)
        else if ( (LRESULT)wParam == RI_START_TARGET_RETRACTED )
            //Navigation succeeded. errorcode (23)
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
            //Navigator is not running or the configuration is not complete. errorcode(3)
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_StartNavigationOverBCR::foo()
{
    //define struct
    RI_CStartNavigationOverBCR data;
    //copy data
    wcscpy(data.m_BCRFileName,L"tour.bcr");
    //write data in shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_StartNavigationOverBCR_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_STARTNAVIGATIONOVERBCR, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_STARTNAVIGATIONOVERBCR);
/** @ingroup CreateBuddyTracking
 *  Message ID for the Request "CreateBuddyTracking":<BR>
	RI_MESSAGE_CREATEBUDDYTRACKING

	<H3>Write Memory:</H3>
	RI_CreateBuddyTracking_WriteData(LPARAM id, RI_CCreateBuddyTracking data);
	<BR>
	<!--
	<H3>Read Memory:</H3>
	RI_CreateBuddyTracking_ReadData(LPARAM id, RI_CCreateBuddyTracking data);
	<BR>
	-->
	<H3>Return Values for CreateBuddyTracking:</H3>
	RI_NOERROR = create buddy tracking succeeded<BR>
    RI_NOTFOUND = create buddy tracking not succeeded, no icon found<BR>
    RI_NOTADDED = create buddy tracking not succeeded, could not create buddy<BR><BR>

	<h2>Description:</h2>create a buddy tracking icon.<br> The file must be present in the oem-directory of the navigation software and the bitmap filename has to be given without path extensions, e.g. don't use "/truck/oem/car.png" but "car.png".<br>
	<br>

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_CreateBuddyTracking::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_CREATEBUDDYTRACKING )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //create buddy tracking succeeded. errorcode (0)
            RI_CCreateBuddyTracking data;
			LRESULT read_suc = RI_CreateBuddyTracking_ReadData( lParam, data );
			//read m_BuddyID
        }
        else if ( (LRESULT)wParam == RI_NOTFOUND)
        {
            //No Icon found
        }
        else if ( (LRESULT)wParam == RI_NOTADDED)
        {
            //could not create buddy
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
        {
            //Navigator is not running or the configuration is not complete. errorcode(3)
        }
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_CreateBuddyTracking::foo()
{
    //define struct
    RI_CCreateBuddyTracking data;
    //copy data
    wcscpy(data.m_BMPFileName,L"car.png");
    //write data in shared memory
    LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_CreateBuddyTracking_WriteData( id, data );
    if ( sharing_ret != RI_NOERROR ) 
        return;
    if ( RI_MESSAGE( RI_MESSAGE_CREATEBUDDYTRACKING, GetSafeHwnd(), id ) == RI_NOERROR )
        //message sending succeeded
    else
        //navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_CREATEBUDDYTRACKING);
/** @ingroup SetBuddyPosition
 *  Message ID for the Request "SetBuddyPosition":<BR>
	RI_MESSAGE_SETBUDDYPOSITION

	<H3>Write Memory:</H3>
	RI_SetBuddyPosition_WriteData(LPARAM id, RI_CSetBuddyPosition data);
	<BR>
	<H3>Return Values for SetBuddyPosition:</H3>
	RI_NOERROR = set buddy position succeeded<BR>
    RI_NOTADDED = set buddy position not succeeded, because the coordinate is not valid.<BR>
    RI_NOTAV = set buddy position not succeeded, could not set coordinate for this buddyId.<BR><BR>

	<h2>Description:</h2>set the position for a buddy tracking icon.<br>
	<br>

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_SetBuddyPosition::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_SETBUDDYPOSITION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //set buddy position succeeded. errorcode (0)
        }
        else if ( (LRESULT)wParam == RI_NOTAV)
        {
            //could not set position, for this buddyID
        }
        else if ( (LRESULT)wParam == RI_NOTADDED)
        {
            //the coordinate is not valid.
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
        {
            //Navigator is not running or the configuration is not complete. errorcode(3)
        }
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_SetBuddyPosition::foo()
{
    //define struct
    RI_CSetBuddyPosition data;
	data.m_BuddyID = 1;
    data.m_Mercator_x = 937185;
    data.m_Mercator_y = 6270130;
	//write data in shared memory
	LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_SetBuddyPosition_WriteData( id, data );
	if ( sharing_ret != RI_NOERROR ) 
        return;
	if ( RI_MESSAGE( RI_MESSAGE_SETBUDDYPOSITION, GetSafeHwnd(), id ) == RI_NOERROR )
		//message sending succeeded
	else
		//navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_SETBUDDYPOSITION);
/** @ingroup DeleteBuddyTracking
 *  Message ID for the Request "DeleteBuddyTracking":<BR>
	RI_MESSAGE_DELETEBUDDYTRACKING

	<H3>Write Memory:</H3>
	RI_DeleteBuddyTracking_WriteData(LPARAM id, RI_CDeleteBuddyTracking data);
	<BR>
	<H3>Return Values for DeleteBuddyTracking:</H3>
	RI_NOERROR = Delete buddy succeeded.<BR>
    RI_NOTAV = Could not delete buddy, because buddyId is not available.<BR><BR>

	<h2>Description:</h2>Delete the buddy tracking icon.<br>
	<br>

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_DeleteBuddyTracking::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_DELETEBUDDYTRACKING )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //delete buddy tracking succeeded. errorcode (0)
        }
        else if ( (LRESULT)wParam == RI_NOTAV)
        {
            //could not delete buddy tracking, because buddyId is not available
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
        {
            //Navigator is not running or the configuration is not complete. errorcode(3)
        }
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_DeleteBuddyTracking::foo()
{
    //define struct
    RI_CDeleteBuddyTracking data;
	data.m_BuddyID = 1;
	//write data in shared memory
	LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_DeleteBuddyTracking_WriteData( id, data );
	if ( sharing_ret != RI_NOERROR ) 
        return;
	if ( RI_MESSAGE( RI_MESSAGE_DELETEBUDDYTRACKING, GetSafeHwnd(), id ) == RI_NOERROR )
		//message sending succeeded
	else
		//navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_DELETEBUDDYTRACKING);
/** @ingroup CurrentFuelConsumption
 *  Message ID for the Request "CurrentFuelConsumption":<BR>
	RI_MESSAGE_CURRENTFUELCONSUMPTION

	<H3>Write Memory:</H3>
	RI_CurrentFuelConsumption_WriteData(LPARAM id, RI_CCurrentFuelConsumption data);
	<BR>
	<H3>Return Values for CurrentFuelConsumption:</H3>
	RI_NOERROR = Set current fuel consumption succeeded<BR>
    RI_NOTRUN = Navigation ist not running.<BR><BR>

	<h2>Description:</h2>Set current fuel consumption.<br>
	<br>

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_CurrentFuelConsumption::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_CURRENTFUELCONSUMPTION )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //set current fuel consumption succeeded. errorcode (0)
        }
        else if ( (LRESULT)wParam == RI_NOTRUN)
        {
            //could not set current fuel consumption, because navigation ist not running
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
        {
            //Navigator is not running or the configuration is not complete. errorcode(3)
        }
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_CurrentFuelConsumption::foo()
{
    //define struct
    RI_CCurrentFuelConsumption data;
	data.m_FuelConsumption = 6;
	//write data in shared memory
	LPARAM id = GetUniqueID();
    LRESULT sharing_ret = RI_CurrentFuelConsumption_WriteData( id, data );
	if ( sharing_ret != RI_NOERROR ) 
        return;
	if ( RI_MESSAGE( RI_MESSAGE_CURRENTFUELCONSUMPTION, GetSafeHwnd(), id ) == RI_NOERROR )
		//message sending succeeded
	else
		//navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_CURRENTFUELCONSUMPTION);
/** @ingroup ReRoutingOccurred
 *  Message ID for the Request "ReRoutingOccurred":<BR>
	RI_MESSAGE_REROUTINGOCCURRED

	<H3>Write Memory:</H3>
	RI_ReRoutingOccurred_WriteData(LPARAM id, RI_CReRoutingOccurred data);
	<BR>
	<H3>Return Values for ReRoutingOccurred:</H3>
	RI_NOERROR = ReRoutingOccurred succeeded<BR>

	<h2>Description:</h2>return the number og reroutings.<br>
	<br>

	<H2>Example:</H2>
\code
#define USE_LOADLIBRARY
#include "TNSRemoteInterfaceDll.h"

LPARAM GetUniqueID()
{
    //generate unique ID
    static LPARAM id = 0;
    if (++id == 0) ++id; // do not use 0 !!!!!
    return id;
}

inline LRESULT RI_MESSAGE( const UINT request, HWND h_client, LPARAM id )
{
    //check if RI handle is valid
    if ( !IsWindow( RI_GetTNS() ) ) 
        return RI_NAVIGATIONNOTACTIVE;
    //send the request
    PostMessage( RI_GetTNS(), request, WPARAM(h_client), id );
    return RI_NOERROR;
}

LRESULT CDlgMFC_ReRoutingOccurred::WindowProc ( UINT  message , WPARAM  wParam , LPARAM  lParam )
{
    //check messages for answer from RI
    if ( message == RI_MESSAGE_REROUTINGOCCURRED )
    {
        if ( (LRESULT)wParam == RI_NOERROR )
        {
            //ReRoutingOccurred succeeded. errorcode (0)
            RI_CReRoutingOccurred data;
			LRESULT read_suc = RI_ReRoutingOccurred_ReadData( lParam, data );
			//read m_NumberOfReRoutings
        }
        else if ( (LRESULT)wParam == RI_NAVIGATIONNOTACTIVE)
        {
            //Navigator is not running or the configuration is not complete. errorcode(3)
        }
    }
    return CDialog::WindowProc( message, wParam, lParam );
}

void CDlgMFC_ReRoutingOccurred::foo()
{
    LPARAM id = GetUniqueID();
    if ( RI_MESSAGE( RI_MESSAGE_REROUTINGOCCURRED, GetSafeHwnd(), id ) == RI_NOERROR )
		//message sending succeeded
	else
		//navigation software not running
}
\endcode
*/
REGISTER_RIMESSAGE( RI_MESSAGE_REROUTINGOCCURRED);
}

// *********************************************************************************
//  Return values
// ********************************************************************************

/*! \namespace */ 
namespace {
/*!
	\ingroup ReturnValues
	RI_NOERROR = 0 when no error occured
*/  
	
const LRESULT RI_NOERROR			=  0;	
/*!
	\ingroup ReturnValues
	RI_FUNCTIONNOTLOADED = 1 when the function could not be loaded with GetProcAddress.
							 Only when USE_LOADLIBRARY is set.
*/
const LRESULT RI_FUNCTIONNOTLOADED	=  1; 
/*!
	\ingroup ReturnValues
	RI_NOSHAREDACCESS = 2 when locking did not work.
						  Should never be raised.
*/
const LRESULT RI_NOSHAREDACCESS		=  2;
/*!
	\ingroup ReturnValues
	RI_NAVIGATIONNOTACTIVE = 3 when navigator not active, or the configuration of the navigator is not completed.
*/ 
const LRESULT RI_NAVIGATIONNOTACTIVE  =  3;
/*!
	\ingroup ReturnValues
	RI_NOTAWINDOW = 4 when not window
*/ 
const LRESULT RI_NOTAWINDOW			=  4;
/*!
	\ingroup ReturnValues
	RI_NOTFOUND = 5 when not found
*/ 
const LRESULT RI_NOTFOUND			=  5;
/*!
	\ingroup ReturnValues
	RI_NOFREEBUFFER = 6 when no free buffer
*/ 
const LRESULT RI_NOFREEBUFFER		=  6;
/*!
	\ingroup ReturnValues
	RI_WRONGDLLVERSION = 7 when DLL version is wrong
*/ 
const LRESULT RI_WRONGDLLVERSION    =  7;
/*!
	\ingroup ReturnValues
	RI_NOTLINKABLE = 10 when Mercator coordinates not linkable for RI_LinkMercator2Map
*/ 
const LRESULT RI_NOTLINKABLE		= 10; //for RI_LinkMercator2Map
/*!
	\ingroup ReturnValues
	RI_NOTADDED = 11 when Stop Off Point not added
*/ 
const LRESULT RI_NOTADDED			= 11; //for RI_StOP_Add
/*!
	\ingroup ReturnValues
	RI_NODESC = 12 when description failed for add stop off point
*/ 
const LRESULT RI_NODESC				= 12; 
/*!
	\ingroup ReturnValues
	RI_ALREADY = 13 when GPS Recording already is running
*/ 
const LRESULT RI_ALREADY			= 13;
/*!
	\ingroup ReturnValues
	RI_TOBIG = 14 when output file for GPS Recording is to big
*/ 
const LRESULT RI_TOBIG			    = 14;
/*!
	\ingroup ReturnValues
	RI_NOTWRITE = 15 when GPS Record could not write to File
*/ 
const LRESULT RI_NOTWRITE			= 15;
/*!
	\ingroup ReturnValues
	RI_STOPGPS = 16 when GPS Records stops
*/ 
const LRESULT RI_STOPGPS			= 16;
/*!
	\ingroup ReturnValues
	RI_NOTRUN = 17 when GPS Record not run
*/ 
const LRESULT RI_NOTRUN				= 17;
/*!
	\ingroup ReturnValues
	RI_NOFIX = 18 when GPS Signal not fix
*/ 
const LRESULT RI_NOFIX				= 18;
/*!
	\ingroup ReturnValues
	RI_NOTAV = 19 when GPS signal not available
*/ 
const LRESULT RI_NOTAV				= 19;
/*!
	\ingroup ReturnValues
	RI_USER = 20 when User abort
*/ 
const LRESULT RI_USER				= 20;

/*!
	\ingroup ReturnValues
	RI_START_RETRACTED = 21 when Startpoint was retracted
*/ 
const LRESULT RI_START_RETRACTED	= 21;
/*!
	\ingroup ReturnValues
	RI_TARGET_RETRACTED = 22 when Target point was retracted
*/ 
const LRESULT RI_TARGET_RETRACTED	= 22;
/*!
	\ingroup ReturnValues
	RI_START_TARGET_RETRACTED = 23 when Start and Target point was retracted
*/ 
const LRESULT RI_START_TARGET_RETRACTED	= 23;

}

// *********************************************************************************
//  Data structures
// ********************************************************************************

/*!
	\ingroup SearchAddress
	\brief Data structure for the Request "SearchAddress".
*/
// parameter struct for RI_CSearchAddress
struct RI_CSearchAddress
{
	LPARAM m_ID;							///< ID used as ACT
	/// Address to search, allowed formats:<BR>
	/// -country, zip code, city, street, housenr<BR>
	/// -mercator (937575,6269540)<BR>
	/// -geodecimal (8.418774,49.009233)
	wchar_t m_address[256];				    ///< Input: address string in format land,postcode,city,street,house nr.
	wchar_t m_result_address[256];			///< Output: address string in format land,postcode,city,street,house nr.
	int m_mercator_x;						///< Output: x-coordinate of linked mercator position
	int m_mercator_y;						///< Output: y-coordinate of linked mercator position
};

/*!
	\ingroup LinkMercator2Map
	\brief Data structure for the Request "link mercator to map".
*/
// parameter struct for RI_CLinkMercator2Map
struct RI_CLinkMercator2Map
{
	LPARAM m_ID;							///< ID used as ACT
	int m_mercator_x;						///< Input: x-coordinate of mercator position to link
	int m_mercator_y;						///< Input: y-coordinate of mercator position to link
	int m_linked_mercator_x;				///< Output: x-coordinate of linked mercator position
	int m_linked_mercator_y;				///< Output: y-coordinate of linked mercator position
    wchar_t m_MapName [256];				///< Output: map of linked mercator position
};

// parameter struct for RI_CAddStopOffPoint
// StOP = Stop off point
/*!
	\ingroup AddStopOffPoint
	\brief Data structure for the Request "AddStopOffPoint".
*/
struct RI_CAddStopOffPoint
{
	LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Caption[128];				    ///< Input: caption of the mercator position
	wchar_t m_Description[128];			    ///< Input: description of the mercator position
	long m_mercator_x;						///< Input: x-coordinate of mercator position to link
	long m_mercator_y;						///< Input: y-coordinate of mercator position to link
};
// parameter struct for RI_CAddStopOffPoint2FavoriteList
/*!
	\ingroup AddStopOffPoint2FavoriteList
	\brief Data structure for the Request "AddStopOffPoint2FavoriteList".
*/
struct RI_CAddStopOffPoint2FavoriteList
{
	LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Caption[128];				    ///< Input: caption of the mercator position
	wchar_t m_Description[128];			    ///< Input: description of the mercator position
	long m_mercator_x;						///< Input: x-coordinate of mercator position to link
	long m_mercator_y;						///< Input: y-coordinate of mercator position to link
};
// parameter struct for RI_CAddStopOffPoint2PreviousDestinationList 
/*!
	\ingroup AddStopOffPoint2PreviousDestinationList
	\brief Data structure for the Request "AddStopOffPoint2PreviousDestinationList".
*/
struct RI_CAddStopOffPoint2PreviousDestinationList
{
	LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Caption[128];				    ///< Input: caption of the mercator position
	wchar_t m_Description[128];			    ///< Input: description of the mercator position
	long m_mercator_x;						///< Input: x-coordinate of mercator position to link
	long m_mercator_y;						///< Input: y-coordinate of mercator position to link
};
// parameter struct for RI_CGetStopOffPointList
/*!
	\ingroup GetStopOffPointList
	\brief Data structure for the Request "GetStopOffPointList".
*/
struct RI_CGetStopOffPointList
{
    LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Caption[128];				    ///< Output: caption of the mercator position
	wchar_t m_Description[128];			    ///< Output: description of the mercator position
	int m_mercator_x;						///< Output: x-coordinate of mercator position to link
	int m_mercator_y;						///< Output: y-coordinate of mercator position to link
	int m_listcount;						///< Output: number of entries
    int m_StationID;						///< Output: ID of the stop off point
	int m_index;							///< Input/Output: current number of the stop off point
	/*!
	Void=0<br>
	GPS=1<br>
	ClickPoint=2<br>
	Addr=3<br>
	HomeBase=4<br>
	PreviousTargets=5<br>
	Favorite=6<br>
	NearestSearch=7<br>
	Poi=8<br>
	CL=9<br> 	
	RC=10<br>  
	BCR=11<br>
	BCRList=12<br>
	PrevTownSearch=13<br>
	PrevStreetSearch=14<br>
	*/
	int m_type;								///< Output: station type
};
// parameter struct for RI_CGetFavoriteList
/*!
	\ingroup GetFavoriteList
	\brief Data structure for the Request "GetFavoriteList".
*/
struct RI_CGetFavoriteList
{
    LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Caption[128];				    ///< Output: caption of the mercator position
	wchar_t m_Description[128];			    ///< Output: description of the mercator position
	int m_mercator_x;						///< Output: x-coordinate of mercator position to link
	int m_mercator_y;						///< Output: y-coordinate of mercator position to link
	int m_listcount;						///< Output: number of entries
    int m_StationID;						///< Output: ID of the stop off point
	int m_index;							///< Input/Output: current number of the stop off point
	/*!
	Void=0<br>
	GPS=1<br>
	ClickPoint=2<br>
	Addr=3<br>
	HomeBase=4<br>
	PreviousTargets=5<br>
	Favorite=6<br>
	NearestSearch=7<br>
	Poi=8<br>
	CL=9<br> 	
	RC=10<br>  
	BCR=11<br>
	BCRList=12<br>
	PrevTownSearch=13<br>
	PrevStreetSearch=14<br>
	*/
	int m_type;								///< Output: station types 
};
// parameter struct for RI_CClearStopOffPoint
/*!
	\ingroup ClearStopOffPoint
	\brief Data structure for the Request "ClearStopOffPoint".
*/
struct RI_CClearStopOffPoint
{
	LPARAM m_ID;							///< ID used as ACT
	int m_StationID;						///< Input: ID of the stop off point
};
// parameter struct for RI_CClearStopOffPointFromFavoriteList
/*!
	\ingroup ClearStopOffPointFromFavoriteList
	\brief Data structure for the Request "ClearStopOffPointFromFavoriteList".
*/
struct RI_CClearStopOffPointFromFavoriteList
{
	LPARAM m_ID;							///< ID used as ACT
	int m_StationID;						///< Input: ID of the stop off point
};
// parameter struct for RI_CGetPreviousDestinationList
/*!
	\ingroup GetPreviousDestinationList
	\brief Data structure for the Request "GetPreviousDestinationList".
*/
struct RI_CGetPreviousDestinationList
{
    LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Caption[128];				    ///< Output: caption of the mercator position
	wchar_t m_Description[128];			    ///< Output: description of the mercator position
	int m_mercator_x;						///< Output: x-coordinate of mercator position to link
	int m_mercator_y;						///< Output: y-coordinate of mercator position to link
	int m_listcount;						///< Output: number of entries
    int m_StationID;						///< Output: ID of the stop off point
	int m_index;							///< Input/Output: current number of the stop off point
	/*!
	Void=0<br>
	GPS=1<br>
	ClickPoint=2<br>
	Addr=3<br>
	HomeBase=4<br>
	PreviousTargets=5<br>
	Favorite=6<br>
	NearestSearch=7<br>
	Poi=8<br>
	CL=9<br> 	
	RC=10<br>  
	BCR=11<br>
	BCRList=12<br>
	PrevTownSearch=13<br>
	PrevStreetSearch=14<br>
	*/
	int m_type;								///< Output: station types
};
// parameter struct for RI_CClearStopOffPointFromPreviousDestinationList
/*!
	\ingroup ClearStopOffPointFromPreviousDestinationList
	\brief Data structure for the Request "ClearStopOffPointFromPreviousDestinationList".
*/
struct RI_CClearStopOffPointFromPreviousDestinationList
{
	LPARAM m_ID;							///< ID used as ACT
	int m_StationID;						///< Input: ID of the stop off point
};
// parameter struct for RI_CActualApplicationState
/*!
	\ingroup ActualApplicationState
	\brief Data structure for the Request "ActualApplicationState".
*/
struct RI_CActualApplicationState
{
	LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Description[128];			    ///< Output: description of the application state
};
// parameter struct for RI_CSystemInfo
/*!
	\ingroup SystemInfo
	\brief Data structure for the Request "SystemInfo".
*/
struct RI_CSystemInfo
{
	LPARAM m_ID;							///< ID used as ACT
	wchar_t m_Version[25];					///< Output: program version number
    wchar_t m_Build[25];					///< Output: program build number
	wchar_t m_CurrentMapName[128];			///< Output: name of the current map
	wchar_t m_CurrentMapDate[10];			///< Output: date of the current map
	wchar_t m_CurrentMapVersion[10];		///< Output: version number of the current map
	wchar_t m_AvailableMaps[128][50];		///< Output: all installed maps 
	wchar_t m_CurrentLanguage[50];			///< Output: current language
	wchar_t m_AvailableLanguages[300];		///< Output: all installed languages
	int m_MapCount;							///< Internal: number of maps

};
// parameter struct for RI_CSwitchMap
/*!
	\ingroup SwitchMap
	\brief Data structure for the Request "SwitchMap".
*/
struct RI_CSwitchMap
{
    LPARAM m_ID;							///< ID used as ACT
	wchar_t m_MapName[128];				    ///< Input: name of map 
	wchar_t m_CurrentMapName[128];			///< Output: current map name
	wchar_t m_Currentmapdate[10];			///< Output: date of the current map
	wchar_t m_CurrentMapversion[10];		///< Output: version of the current map
	wchar_t m_AvailableMaps[128][10];		///< Output: all installed maps
	int m_MapCount;							///< Internal: number of maps

};
// parameter struct for RI_CChangeApplicationOptions
/*!
	\ingroup ChangeApplicationOptions
	\brief Data structure for the Request "ChangeApplicationOptions".
*/
struct RI_CChangeApplicationOptions
{   
	LPARAM m_ID;							///< ID used as ACT
	int m_InfoLeftSide;						///< Input: toggle the information shown in OptInfo1
	int m_InfoRightSide;					///< Input: toggle the information shown in OptInfo2
	/*!		
	RouteOptimisation value can be one of the following types:<br>
	RouteOptShort = 0<br>
	RouteOptEconomic = 1<br>
	RouteOptFast = 2<br>
	*/
	int m_Optimisation;						///< Input: change route optimisation (0..2)
	int m_Volume;							///< Input: change volume range (0..100)
	/*!		
	Vehicle value can be one of the following types:<br>
	VehicleCarFast = 0 <br>
	VehicleCarSlow = 1 <br>
	VehiclePedestrian = 2<br>
	VehicleCyclist = 3<br>
	VehicleMotorcyclist = 4<br>
	VehicleCaravan = 5<br>
	VehicleTruckSlow = 6<br>
	VehicleTruckFast = 7<br>
	*/
	int m_Vehicle;							///< Input: change vehicle profile (0..7)
	bool m_ShowStreetNames;					///< Input: show street name (0 or 1)
	bool m_ShowCompass;						///< Input:	show compass (0 or 1)
	bool m_ShowScale;						///< Input:	show scale (0 or 1)
	bool m_MapFacingNorth;					///< Input: set map facing north (0 or 1)
	bool m_DependentOnSpeed;				///< Input: activate autozoom (0 or 1)
	bool m_AvoidUTurns;						///< Input: avoid u-turns (0 or 1)
	bool m_MapInItinerary;					///< Input: set routelist northen adjusted (0 or 1)
	bool m_Motorway;						///< Input: use motorways (0 or 1)
	bool m_Ferries;							///< Input: use ferries (0 or 1)
};
// parameter struct for RI_CStartGPSRecord
/*!
	\ingroup StartGPSRecord
	\brief Data structure for the Request "StartGPSRecord".
*/
struct RI_CStartGPSRecord
{
    LPARAM m_ID;							///< ID used as ACT
	wchar_t m_FileName[128];				///< Input: name of the log file (optional, default = "gpsrecord.out")
	int m_MaxFileSize;						///< Input: max filesize (optional)
	int m_Interval;							///< Input: interval for writing GPS position in sec. (optional, default = 10) 
	/*!
	0 = Geodec
	1 = Mercator
	*/
	int m_OutputFormat;						///< Input: the coordinate format of the GPS position (0 or 1) (optional, default = 1) 
};
// parameter struct for RI_CStopGPSRecord
/*!
	\ingroup StopGPSRecord
	\brief Data structure for the Request "StopGPSRecord".
*/
struct RI_CStopGPSRecord
{
    LPARAM m_ID;							///< ID used as ACT
	int m_NumberEntries;					///< Output: number of entries in the log
	int m_FileSize;							///< Output: filesize of the log
};
// parameter struct for RI_CShowMessage
/*!
	\ingroup ShowMessage
	\brief Data structure for the Request "ShowMessage".
*/
struct RI_CShowMessage
{
    LPARAM m_ID;							///< ID used as ACT
	unsigned int m_MessageButton;			///< Input: message buttons to show (ok=1, ok/cancel=2, yes/no=3)
	unsigned int m_Feedback;				///< Output: button pressed from the user (ok=1, cancel=2, yes=3, no=4)
	wchar_t m_MessageCaption[128];			///< Input: caption of the message
	wchar_t m_Message[256];				    ///< Input: message
};
// parameter struct for RI_CRoutePlanning
/*!
	\ingroup RoutePlanning
	\brief Data structure for the Request "RoutePlanning".
*/
struct RI_CRoutePlanning
{
	LPARAM m_ID;							///< ID used as ACT
	int m_ListCount;						///< Output: number of maneuver
	wchar_t m_ManeuverText[128];			///< Output: maneuver text
    int m_Retracted;                        ///< Output: was start/target retracted
	int m_Road;								///< Output: distance from start to this stop off point in meters
	int m_Duration;							///< Output: estimated duration from start to this stop off point in sec
	int m_Direction;						///< Output: the actual direction in degrees
    int m_mercator_x;						///< Output: the mercator x-coordinates for this stop off point
	int m_mercator_y;						///< Output: the mercator y-coordinates for this stop off point
	int m_TotalTime;						///< Output: the estimated duration from start to end
	int m_TotalDistance;					///< Output: the distance from start to end in meters
    int m_index;							///< Input/Output: actual number of maneuver
	bool m_flag;							///< Internal: flag for the state machine. Don't Edit!!
};
// parameter struct for RI_CInverseGeoCoding
/*!
	\ingroup InverseGeoCoding
	\brief Data structure for the Request "InverseGeoCoding".
*/
struct RI_CInverseGeoCoding
{
	LPARAM m_ID;							///< ID used as ACT
	wchar_t m_address[256];				    ///< Output: address string (country,zip code,city,street,housenumber)
	int m_mercator_x;						///< Input: x-coordinate of mercator position
	int m_mercator_y;						///< Input: y-coordinate of mercator position
};
// parameter struct for RI_CGPSInfo
/*!
	\ingroup GPSInfo
	\brief Data structure for the Request "GPSInfo".
*/
struct RI_CGPSInfo
{
	LPARAM m_ID;							///< ID used as ACT
	double n_geodec_x;						///< Output: actual roadmatched x-position in geographical notation
	double n_geodec_y;						///< Output: actual roadmatched y-position in geographical notation
	double m_geodec_x;						///< Output: actual GPS x-position in geographical notation
	double m_geodec_y;						///< Output: actual GPS y-position in geographical notation
	int m_mercator_x;						///< Output: actual GPS x-position in mercator notation
	int m_mercator_y;						///< Output: actual GPS y-position in mercator notation
	int m_lon;								///< Output: actual GPS x-position in longitude and latitude
	int m_lat;								///< Output: actual GPS y-position in longitude and latitude
	int m_course;							///< Output: actual duration
	int m_velocity;							///< Output: actual speed
	int m_noSatsUsed;						///< Output: number of satellites used for GPS
	int m_Altitude;							///< Output: actual altitude in meters
	int m_lastdata;							///< Output: the time span in msec since the last valid GPS position
	int m_disttotarget;						///< Output: distance in meters to the navigation destination
	int m_timetotarget;						///< Output: estimated time in seconds to reach the navigation destination
	int m_SpeedLimit;						///< Output: actual speedlimit of current segment. if truck profile activated, then returns the actual truck speedlimit of current segment.
	wchar_t m_date [16];					///< Output: actual satellite date
	wchar_t m_time [8];					    ///< Output: actual satellite time
	wchar_t n_mercator[128];				///< Output: actual roadmatched position in mercator notation
	wchar_t m_street [50];					///< Output: name of the street at the actual position
	wchar_t m_target [256];				    ///< Output: target as address string
	bool m_LastDestinationReached;			///< Output: true if last destination reached, else false
	bool flag;								///< Internal: flag for the navigation information section
};
// parameter struct for RI_CGPSLog
/*!
	\ingroup GPSLog
	\brief Data structure for the Request "GPSLog".
*/
struct RI_CGPSLog
{
	LPARAM m_ID;							///< ID used as ACT			
	wchar_t m_Nmea[512];					///< Input: NMEA Data		
	
};
// parameter struct for RI_CRefreshMap
/*!
	\ingroup RefreshMap
	\brief Data structure for the Request "RefreshMap".
*/
struct RI_CRefreshMap
{
	LPARAM m_ID;							///< ID used as ACT	
	/*!
	0 = Disable <br>
	1 = Enable	<br>
	*/
	bool m_refresh;							///< Input: enable/disable map update	
	
};
// parameter struct for RI_CNaviControls
/*!
	\ingroup NaviControls
	\brief Data structure for the Request "NaviControls".
*/
struct RI_CNaviControls
{
	LPARAM m_ID;							///< ID used as ACT
	/*!
	-1 = do nothing<br>
	0 = ZoomIn <br>
	1 = ZoomOut <br>
	2 = SkipStation <br>
	*/
	int m_value;							///< Input: emulate click event on NaviControl-Buttons ZoomIn, ZoomOut, SkipStation
};
// parameter struct for RI_CSetMapCenter
/*!
	\ingroup SetMapCenter
	\brief Data structure for the Request "SetMapCenter".
*/
struct RI_CSetMapCenter
{
	LPARAM m_ID;							///< ID used as ACT				
	bool m_panning;							///< Input: enable/disable
	/*!
	Mercator = true<br>
	Geodec = false<br>
   */
	bool m_flag;							///< Input: flag for input format
	wchar_t m_PosX[20];					    ///< Input: x-coordinate
	wchar_t m_PosY[20];					    ///< Input: y-coordinate
	
};
// parameter struct for RI_CVisibleMapRect
/*!
	\ingroup VisibleMapRect
	\brief Data structure for the Request "VisibleMapRect".
*/
struct RI_CVisibleMapRect
{
	LPARAM m_ID;							///< ID used as ACT				
	bool m_panning;							///< Input: enable/disable
	/*!
	Mercator = true<br>
	Geodec = false<br>
	*/
	bool m_flag;							///< Input: flag for input format
	wchar_t  m_map_bottom[20];				///< Input: MapRect.Bottom
	wchar_t  m_map_right[20];				///< Input: MapRect.Right
	wchar_t  m_map_left[20];				///< Input: MapRect.Left
	wchar_t  m_map_top[20];				    ///< Input: MapRect.Top
};
// parameter struct for RI_CRiDump
/*!
	\ingroup RiDump
	\brief Data structure for the Request "RiDump".
*/
struct RI_CRiDump
{
	LPARAM m_ID;							///< ID used as ACT				
	bool m_dump;							///< Input: enable/disable dump	
	
};
// parameter struct for RI_CSetProfile
/*!
	\ingroup SetProfile
	\brief Data structure for the Request "SetProfile".
*/
struct RI_CSetProfile
{
	LPARAM m_ID;							///< ID used as ACT				
	wchar_t  m_ProfileName[256];			///< Input: profilename
};
// parameter struct for RI_CGetProfile
/*!
	\ingroup GetProfile
	\brief Data structure for the Request "GetProfile".
*/
struct RI_CGetProfile
{
	LPARAM m_ID;							///< ID used as ACT				
	wchar_t  m_ProfileName[256];			///< Output: profilename
};
// parameter struct for RI_CSetDayNight
/*!
	\ingroup SetDayNight
	\brief Data structure for the Request "SetDayNight".
*/
struct RI_CSetDayNight
{
	LPARAM m_ID;							///< ID used as ACT
	/*!
	true = Day<br>
	false = Night<br>
	*/
	bool  m_bDayNight;          			///< Input: set day mode 
};
// parameter struct for RI_CGetDayNight
/*!
	\ingroup GetDayNight
	\brief Data structure for the Request "GetDayNight".
*/
struct RI_CGetDayNight
{
	LPARAM m_ID;							///< ID used as ACT
	/*!
	true = Day<br>
	false = Night<br>
	*/
	bool  m_bDayNight;          			///< Output: get day mode
};
// parameter struct for RI_CStartNavigationOverBCR
/*!
	\ingroup StartNavigationOverBCR
	\brief Data structure for the Request "StartNavigationOverBCR".
*/
struct RI_CStartNavigationOverBCR
{
	LPARAM m_ID;							///< ID used as ACT				
	wchar_t m_BCRFileName [260];			///< Input: BCRFileName without path, e.g. "foo.bcr"
};

// parameter struct for RI_CCreateBuddyTracking
/*!
	\ingroup CreateBuddyTracking
	\brief Data structure for the Request "CreateBuddyTracking".
*/
struct RI_CCreateBuddyTracking
{
	LPARAM m_ID;							///< ID used as ACT				
	wchar_t m_BMPFileName [260];			///< Input: BMPFileName without path, e.g. "car.png"
    int m_BuddyID;                          ///< Output: ID of the buddy

};

// parameter struct for RI_CSetBuddyPosition
/*!
	\ingroup SetBuddyPosition
	\brief Data structure for the Request "SetBuddyPosition".
*/
struct RI_CSetBuddyPosition
{
	LPARAM m_ID;							///< ID used as ACT				
	int m_BuddyID;                          ///< Input: ID of the buddy
    int m_Mercator_x;                       ///< Input: the mercator x-coordinates for this Buddy
    int m_Mercator_y;                       ///< Input: the mercator y-coordinates for this Buddy

};

// parameter struct for RI_CDeleteBuddyTracking
/*!
	\ingroup DeleteBuddyTracking
	\brief Data structure for the Request "DeleteBuddyTracking".
*/
struct RI_CDeleteBuddyTracking
{
	LPARAM m_ID;							///< ID used as ACT				
	int m_BuddyID;                          ///< Input: ID of the buddy
};
// parameter struct for RI_CCurrentFuelConsumption
/*!
	\ingroup CurrentFuelConsumption
	\brief Data structure for the Request "CurrentFuelConsumption".
*/
struct RI_CCurrentFuelConsumption
{
    LPARAM m_ID;                        ///< ID used as ACT	
    double m_FuelConsumption;           ///< Input: current fuel consumption
};
// parameter struct for RI_CReRoutingOccurred
/*!
	\ingroup ReRoutingOccurred
	\brief Data structure for the Request "ReRoutingOccurred".
*/
struct RI_CReRoutingOccurred
{
    LPARAM m_ID;                        ///< ID used as ACT	
    int m_NumberOfReRoutings;           ///< Output: number of reroutings
};
// *********************************************************************************
//  Dynamic library loading
// ********************************************************************************
#ifdef USE_LOADLIBRARY

class CLoadDLL
{
	public:
		explicit CLoadDLL( LPCTSTR name )
		{
			dll_instance = LoadLibrary( name );
		}
		~CLoadDLL()
		{
			if ( dll_instance ) FreeLibrary( dll_instance );
		}
		HINSTANCE get() const
		{
			return dll_instance;
		}
	private:
		HINSTANCE dll_instance;
};

class CRemoteDLL
{
	public:
		static HINSTANCE getDLL()
		{
			static CLoadDLL dll_instance( TEXT("ri.dll") );
			return dll_instance.get();
		}
};

#endif // USE_LOADLIBRARY


// *********************************************************************************
//  Function definitions/declarations 
// ********************************************************************************

extern "C" 
{
DLL_FUNC_1( RI_RegisterTNS,		BOOL,			FALSE,		HWND ) // as a client, never use this function !!!
DLL_FUNC_0( RI_UnregisterTNS,	BOOL,			FALSE ) // as a client, never use this function !!!
/*!
	\ingroup MessageSystem
	\brief Get the message receiver handle of the navigation software

    Returns the message receiver handle of the navigation software. 
	This handle cannot be used for purposes other than RI-handling because it is not the 
	handle of the navigation application or the navigator main window.
    \return handle to message receiver in navigator
 */
DLL_FUNC_0( RI_GetTNS,			HWND,			NULL )
/*!
	\ingroup VersionControl
	\brief Get the version number of the dll.

	\return dll version number
*/
DLL_FUNC_0( RI_GetDLLVersion,	unsigned long,	0 )

/*!
	\ingroup LinkMercator2Map
	\brief Read data from shared memory for "LinkMercator2Map".
*/
REQ_FUNC_2( RI_LinkMercator2Map_ReadData,					RI_CLinkMercator2Map & )
/*!
	\ingroup LinkMercator2Map
	\brief Write data to shared memory for "LinkMercator2Map".
*/
REQ_FUNC_2( RI_LinkMercator2Map_WriteData,					RI_CLinkMercator2Map & )
/*!
	\ingroup SearchAddress
	\brief Read data from shared memory for "SearchAddress".
*/
REQ_FUNC_2( RI_SearchAddress_ReadData,						RI_CSearchAddress & )
/*!
	\ingroup SearchAddress
	\brief Write data to shared memory for "SearchAddress".
*/
REQ_FUNC_2( RI_SearchAddress_WriteData,						RI_CSearchAddress & )
// ... control the stop-off point list
/*!
	\ingroup AddStopOffPoint_internal
	\brief Read data from shared memory for "Add Stop Off Point".
*/
REQ_FUNC_2( RI_AddStopOffPoint_ReadData, RI_CAddStopOffPoint & )
/*!
	\ingroup AddStopOffPoint
	\brief Write data to shared memory for "Add Stop Off Point".
*/
REQ_FUNC_2( RI_AddStopOffPoint_WriteData, RI_CAddStopOffPoint & )
/*!
	\ingroup AddStopOffPoint2FavoriteList_internal
	\brief Read data from shared memory for "Add Stop Off Point 2 Favorite List".
*/
REQ_FUNC_2( RI_AddStopOffPoint2FavoriteList_ReadData, RI_CAddStopOffPoint2FavoriteList & )
/*!
	\ingroup AddStopOffPoint2FavoriteList
	\brief Write data to shared memory for "Add Stop Off Point 2 Favorite List".
*/
REQ_FUNC_2( RI_AddStopOffPoint2FavoriteList_WriteData, RI_CAddStopOffPoint2FavoriteList & )
/*!
	\ingroup AddStopOffPoint2PreviousDestinationList_internal
	\brief Read data from shared memory for "Add Stop Off Point 2 Previous DestinationList".
*/
REQ_FUNC_2( RI_AddStopOffPoint2PreviousDestinationList_ReadData, RI_CAddStopOffPoint2PreviousDestinationList & )
/*!
	\ingroup AddStopOffPoint2PreviousDestinationList
	\brief Write data to shared memory for "Add Stop Off Point 2 Previous DestinationList".
*/
REQ_FUNC_2( RI_AddStopOffPoint2PreviousDestinationList_WriteData, RI_CAddStopOffPoint2PreviousDestinationList & )
/*!
	\ingroup GetStopOffPointList
	\brief Read data from shared memory for "Get Stop Off Point List".
*/
REQ_FUNC_2( RI_GetStopOffPointList_ReadData, RI_CGetStopOffPointList & )
/*!
	\ingroup GetStopOffPointList
	\brief Write data to shared memory for "Get Stop Off Point List".
*/
REQ_FUNC_2( RI_GetStopOffPointList_WriteData, RI_CGetStopOffPointList & )
/*!
	\ingroup GetFavoriteList
	\brief Read data from shared memory for "Get stop off point from FavoriteList".
*/
REQ_FUNC_2( RI_GetFavoriteList_ReadData, RI_CGetFavoriteList & )
/*!
	\ingroup GetFavoriteList
	\brief Write data to shared memory for "Get stop off point from FavoriteList".
*/
REQ_FUNC_2( RI_GetFavoriteList_WriteData, RI_CGetFavoriteList & )
/*!
	\ingroup ClearStopOffPoint
	\brief Read data from shared memory for "Clear Stop Off Point".
*/
REQ_FUNC_2( RI_ClearStopOffPoint_ReadData, RI_CClearStopOffPoint & )
/*!
	\ingroup ClearStopOffPoint
	\brief Write data to shared memory for "Clear Stop Off Point".
*/
REQ_FUNC_2( RI_ClearStopOffPoint_WriteData,	RI_CClearStopOffPoint & )
/*!
	\ingroup ClearStopOffPointFromFavoriteList
	\brief Read data from shared memory for "Clear Stop Off Point From FavoriteList".
*/
REQ_FUNC_2( RI_ClearStopOffPointFromFavoriteList_ReadData,	RI_CClearStopOffPointFromFavoriteList & )
/*!
	\ingroup ClearStopOffPointFromFavoriteList
	\brief Write data to shared memory for "Clear Stop Off Point From FavoriteList".
*/
REQ_FUNC_2( RI_ClearStopOffPointFromFavoriteList_WriteData,	RI_CClearStopOffPointFromFavoriteList & )
/*!
	\ingroup GetPreviousDestinationList
	\brief Read data from shared memory for "Get stop off point from Previous DestinationList".
*/
REQ_FUNC_2( RI_GetPreviousDestinationList_ReadData,	RI_CGetPreviousDestinationList & )
/*!
	\ingroup GetPreviousDestinationList
	\brief Write data to shared memory for "Get stop off point from Previous DestinationList".
*/
REQ_FUNC_2( RI_GetPreviousDestinationList_WriteData, RI_CGetPreviousDestinationList & )
/*!
	\ingroup ClearStopOffPointFromPreviousDestinationList
	\brief Read data from shared memory for "Clear Stop Off Point From Previous DestinationList".
*/
REQ_FUNC_2( RI_ClearStopOffPointFromPreviousDestinationList_ReadData, RI_CClearStopOffPointFromPreviousDestinationList & )
/*!
	\ingroup ClearStopOffPointFromPreviousDestinationList
	\brief Write data to shared memory for "Clear Stop Off Point From Previous DestinationList".
*/
REQ_FUNC_2( RI_ClearStopOffPointFromPreviousDestinationList_WriteData, RI_CClearStopOffPointFromPreviousDestinationList & )
/*!
	\ingroup ActualApplicationState
	\brief Read data from shared memory for "Actual Application State".
*/
REQ_FUNC_2( RI_ActualApplicationState_ReadData, RI_CActualApplicationState & )
/*!
	\ingroup ActualApplicationState_internal
	\brief Write data to shared memory for "Actual Application State".
*/
REQ_FUNC_2( RI_ActualApplicationState_WriteData, RI_CActualApplicationState & )
/*!
	\ingroup SystemInfo
	\brief Read data from shared memory for "System information".
*/
REQ_FUNC_2( RI_SystemInfo_ReadData,	RI_CSystemInfo & )
/*!
	\ingroup SystemInfo_internal
	\brief Write data to shared memory for "System information".
*/
REQ_FUNC_2( RI_SystemInfo_WriteData, RI_CSystemInfo & )
/*!
	\ingroup SwitchMap
	\brief Read data from shared memory for "switch map".
*/
REQ_FUNC_2( RI_SwitchMap_ReadData, RI_CSwitchMap & )
/*!
	\ingroup SwitchMap
	\brief Write data to shared memory for "switch map".
*/
REQ_FUNC_2( RI_SwitchMap_WriteData, RI_CSwitchMap & )
/*!
	\ingroup ChangeApplicationOptions_internal
	\brief Read data from shared memory for "Change Application Options".
*/
REQ_FUNC_2( RI_ChangeApplicationOptions_ReadData, RI_CChangeApplicationOptions & )
/*!
	\ingroup ChangeApplicationOptions
	\brief Write data to shared memory for "Change Application Options".
*/
REQ_FUNC_2( RI_ChangeApplicationOptions_WriteData, RI_CChangeApplicationOptions & )
/*!
	\ingroup StartGPSRecord_internal
	\brief Read data from shared memory for "Start GPS Record".
*/
REQ_FUNC_2( RI_StartGPSRecord_ReadData,	RI_CStartGPSRecord & )
/*!
	\ingroup StartGPSRecord
	\brief Write data to shared memory for "Start GPS Record".
*/
REQ_FUNC_2( RI_StartGPSRecord_WriteData, RI_CStartGPSRecord & )
/*!
	\ingroup StopGPSRecord
	\brief Read data from shared memory for "Stop GPS Record".
*/
REQ_FUNC_2( RI_StopGPSRecord_ReadData, RI_CStopGPSRecord & )
/*!
	\ingroup StopGPSRecord_internal
	\brief Write data to shared memory for "Stop GPS Record".
*/
REQ_FUNC_2( RI_StopGPSRecord_WriteData, RI_CStopGPSRecord & )
/*!
	\ingroup ShowMessage
	\brief Read data from shared memory for "Show Message".
*/
REQ_FUNC_2( RI_ShowMessage_ReadData, RI_CShowMessage & )
/*!
	\ingroup ShowMessage
	\brief Write data to shared memory for "Show Message".
*/
REQ_FUNC_2( RI_ShowMessage_WriteData, RI_CShowMessage & )
/*!
	\ingroup RoutePlanning
	\brief Read data from shared memory for "Route Planning".
*/
REQ_FUNC_2( RI_RoutePlanning_ReadData, RI_CRoutePlanning & )
/*!
	\ingroup RoutePlanning
	\brief Write data to shared memory for "Route Planning".
*/
REQ_FUNC_2( RI_RoutePlanning_WriteData, RI_CRoutePlanning & )
/*!
	\ingroup InverseGeoCoding
	\brief Read data from shared memory for "Inverse GeoCoding".
*/
REQ_FUNC_2( RI_InverseGeoCoding_ReadData, RI_CInverseGeoCoding & )
/*!
	\ingroup InverseGeoCoding
	\brief Write data to shared memory for "Inverse GeoCoding".
*/
REQ_FUNC_2( RI_InverseGeoCoding_WriteData, RI_CInverseGeoCoding & )
/*!
	\ingroup GPSInfo
	\brief Read data from shared memory for "GPS Info".
*/
REQ_FUNC_2( RI_GPSInfo_ReadData, RI_CGPSInfo & )
/*!
	\ingroup GPSInfo_internal
	\brief Write data to shared memory for "GPS Info".
*/
REQ_FUNC_2( RI_GPSInfo_WriteData, RI_CGPSInfo & )
/*!
	\ingroup GPSLog_internal
	\brief Read data from shared memory for "GPS Log".
*/
REQ_FUNC_2( RI_GPSLog_ReadData,	RI_CGPSLog & )
/*!
	\ingroup GPSLog
	\brief Write data to shared memory for "GPS Log".
*/
REQ_FUNC_2( RI_GPSLog_WriteData, RI_CGPSLog & )
/*!
	\ingroup RefreshMap_internal
	\brief Read data from shared memory for "RefreshMap".
*/
REQ_FUNC_2( RI_RefreshMap_ReadData, RI_CRefreshMap & )
/*!
	\ingroup RefreshMap
	\brief Write data to shared memory for "RefreshMap".
*/
REQ_FUNC_2( RI_RefreshMap_WriteData, RI_CRefreshMap & )
/*!
	\ingroup NaviControls_internal
	\brief Read data from shared memory for "NaviControls".
*/
REQ_FUNC_2( RI_NaviControls_ReadData, RI_CNaviControls & )
/*!
	\ingroup NaviControls
	\brief Write data to shared memory for "NaviControls".
*/
REQ_FUNC_2( RI_NaviControls_WriteData, RI_CNaviControls & )
/*!
	\ingroup SetMapCenter_internal
	\brief Read data from shared memory for "SetMapCenter".
*/
REQ_FUNC_2( RI_SetMapCenter_ReadData, RI_CSetMapCenter & )
/*!
	\ingroup SetMapCenter
	\brief Write data to shared memory for "SetMapCenter".
*/
REQ_FUNC_2( RI_SetMapCenter_WriteData, RI_CSetMapCenter & )
/*!
	\ingroup VisibleMapRect_internal
	\brief Read data from shared memory for "VisibleMapRect".
*/
REQ_FUNC_2( RI_VisibleMapRect_ReadData, RI_CVisibleMapRect & )
/*!
	\ingroup VisibleMapRect
	\brief Write data to shared memory for "VisibleMapRect".
*/
REQ_FUNC_2( RI_VisibleMapRect_WriteData, RI_CVisibleMapRect & )
/*!
	\ingroup RiDump
	\brief Read data from shared memory for "RiDump".
*/
REQ_FUNC_2( RI_RiDump_ReadData,	RI_CRiDump & )
/*!
	\ingroup RiDump
	\brief Write data to shared memory for "RiDump".
*/
REQ_FUNC_2( RI_RiDump_WriteData, RI_CRiDump & )
/*!
	\ingroup RiSetProfile_internal
	\brief Read data from shared memory for "RiSetProfile".
*/
REQ_FUNC_2( RI_SetProfile_ReadData,	RI_CSetProfile & )
/*!
	\ingroup RiSetProfile
	\brief Write data to shared memory for "RiSetProfile".
*/
REQ_FUNC_2( RI_SetProfile_WriteData, RI_CSetProfile & )
/*!
	\ingroup RiGetProfile
	\brief Read data from shared memory for "RiGetProfile".
*/
REQ_FUNC_2( RI_GetProfile_ReadData, RI_CGetProfile & )
/*!
	\ingroup RiGetProfile_internal
	\brief Write data to shared memory for "RiGetProfile".
*/
REQ_FUNC_2( RI_GetProfile_WriteData, RI_CGetProfile & )
/*!
	\ingroup RiSetDayNight_internal
	\brief Read data from shared memory for "RiSetDayNight".
*/
REQ_FUNC_2( RI_SetDayNight_ReadData, RI_CSetDayNight & )
/*!
	\ingroup RiSetDayNight
	\brief Write data to shared memory for "RiSetDayNight".
*/
REQ_FUNC_2( RI_SetDayNight_WriteData, RI_CSetDayNight & )
/*!
	\ingroup RiGetDayNight
	\brief Read data from shared memory for "RiGetDayNight".
*/
REQ_FUNC_2( RI_GetDayNight_ReadData, RI_CGetDayNight & )
/*!
	\ingroup RiGetDayNight_internal
	\brief Write data to shared memory for "RiGetDayNight".
*/
REQ_FUNC_2( RI_GetDayNight_WriteData, RI_CGetDayNight & )
/*!
	\ingroup RiStartNavigationOverBCR_internal
	\brief Read data from shared memory for "RiStartNavigationOverBCR".
*/
REQ_FUNC_2( RI_StartNavigationOverBCR_ReadData, RI_CStartNavigationOverBCR & )
/*!
	\ingroup RiStartNavigationOverBCR
	\brief Write data to shared memory for "RiStartNavigationOverBCR".
*/
REQ_FUNC_2( RI_StartNavigationOverBCR_WriteData, RI_CStartNavigationOverBCR & )
/*!
	\ingroup RiCreateBuddyTracking_internal
	\brief Read data from shared memory for "RiCreateBuddyTracking".
*/
REQ_FUNC_2( RI_CreateBuddyTracking_ReadData, RI_CCreateBuddyTracking & )
/*!
	\ingroup RiCreateBuddyTracking
	\brief Write data to shared memory for "RiCreateBuddyTracking".
*/
REQ_FUNC_2( RI_CreateBuddyTracking_WriteData, RI_CCreateBuddyTracking & )

/*!
	\ingroup RiSetBuddyPosition_internal
	\brief Read data from shared memory for "RiSetBuddyPosition".
*/
REQ_FUNC_2( RI_SetBuddyPosition_ReadData, RI_CSetBuddyPosition & )
/*!
	\ingroup RiSetBuddyPosition
	\brief Write data to shared memory for "RiSetBuddyPosition".
*/
REQ_FUNC_2( RI_SetBuddyPosition_WriteData, RI_CSetBuddyPosition & )

/*!
	\ingroup RiDeleteBuddyTracking_internal
	\brief Read data from shared memory for "RiDeleteBuddyTracking".
*/
REQ_FUNC_2( RI_DeleteBuddyTracking_ReadData, RI_CDeleteBuddyTracking & )
/*!
	\ingroup RiDeleteBuddyTracking
	\brief Write data to shared memory for "RiDeleteBuddyTracking".
*/
REQ_FUNC_2( RI_DeleteBuddyTracking_WriteData, RI_CDeleteBuddyTracking & )
/*!
	\ingroup RiCurrentFuelConsumption_internal
	\brief Read data from shared memory for "RiCurrentFuelConsumption".
*/
REQ_FUNC_2( RI_CurrentFuelConsumption_ReadData, RI_CCurrentFuelConsumption & )
/*!
	\ingroup RiCurrentFuelConsumption
	\brief Write data to shared memory for "RiCurrentFuelConsumption".
*/
REQ_FUNC_2( RI_CurrentFuelConsumption_WriteData, RI_CCurrentFuelConsumption & )
/*!
	\ingroup RiReRoutingOccurred_internal
	\brief Read data from shared memory for "RiReRoutingOccurred_internal".
*/
REQ_FUNC_2( RI_ReRoutingOccurred_ReadData, RI_CReRoutingOccurred & )
/*!
	\ingroup RiReRoutingOccurred
	\brief Write data to shared memory for "RiReRoutingOccurred".
*/
REQ_FUNC_2( RI_ReRoutingOccurred_WriteData, RI_CReRoutingOccurred & )
}
#endif